const express = require('express');
const router = express.Router();
const { 
  StatisticsService, 
  ChatSessionService, 
  ApiMappingService, 
  QADatabaseService 
} = require('../services/databaseService');

// Kiểm tra kết nối database
router.get('/health', async (req, res) => {
  try {
    const isConnected = await ApiMappingService.checkConnection();
    res.json({
      success: true,
      data: {
        database: isConnected ? 'connected' : 'disconnected',
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    console.error('❌ Health check failed:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Lấy thống kê tổng quan
router.get('/overview', async (req, res) => {
  try {
    // Lấy thống kê tổng hợp từ tất cả dữ liệu
    const { ChatSession, ChatMessage } = require('../models');
    
    // Đếm tổng sessions
    const totalSessions = await ChatSession.count();
    
    // Đếm tổng messages
    const totalMessages = await ChatMessage.count();
    
    // Đếm API calls
    const apiCalls = await ChatMessage.count({
      where: { message_type: 'api_call' }
    });
    
    // Đếm Q&A responses
    const qaResponses = await ChatMessage.count({
      where: { message_type: 'qa_response' }
    });
    
    // Đếm ChatGPT responses
    const chatgptResponses = await ChatMessage.count({
      where: { message_type: 'bot' }
    });
    
    // Đếm lỗi (messages có error)
    const errors = await ChatMessage.count({
      where: {
        [require('sequelize').Op.or]: [
          { error: { [require('sequelize').Op.ne]: null } },
          { error: { [require('sequelize').Op.ne]: '' } }
        ]
      }
    });
    
    // Tính thời gian phản hồi trung bình
    const avgResponseTime = await ChatMessage.findOne({
      attributes: [
        [require('sequelize').fn('AVG', require('sequelize').col('processing_time')), 'avg_time']
      ],
      where: {
        processing_time: {
          [require('sequelize').Op.not]: null
        }
      }
    });

    res.json({
      success: true,
      data: {
        totalSessions: totalSessions,
        totalMessages: totalMessages,
        apiCalls: apiCalls,
        qaResponses: qaResponses,
        chatgptResponses: chatgptResponses,
        errors: errors,
        avgResponseTime: avgResponseTime ? Math.round(parseFloat(avgResponseTime.getDataValue('avg_time'))) : 0
      }
    });
  } catch (error) {
    console.error('❌ Lỗi lấy thống kê tổng quan:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Lấy thống kê theo ngày (mặc định 7 ngày) - ĐÃ SỬA LỖI
router.get('/daily', async (req, res) => {
  try {
    const days = 7; // Mặc định 7 ngày
    const { ChatSession, ChatMessage } = require('../models');
    const { Op, fn, col } = require('sequelize');
    
    // Tính ngày bắt đầu
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);
    
    // Lấy thống kê theo ngày - sử dụng cú pháp đơn giản
    const dailyStats = await ChatMessage.findAll({
      attributes: [
        [fn('DATE', col('created_at')), 'date'],
        [fn('COUNT', col('id')), 'total_messages']
      ],
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        }
      },
      group: [fn('DATE', col('created_at'))],
      order: [[fn('DATE', col('created_at')), 'ASC']]
    });

    // Lấy thống kê chi tiết theo loại message
    const apiCalls = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        message_type: 'api_call'
      }
    });

    const qaResponses = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        message_type: 'qa_response'
      }
    });

    const chatgptResponses = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        message_type: 'bot'
      }
    });

    const errors = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        error: {
          [Op.ne]: null
        }
      }
    });
    
    // Lấy thống kê sessions theo ngày
    const dailySessions = await ChatSession.findAll({
      attributes: [
        [fn('DATE', col('created_at')), 'date'],
        [fn('COUNT', col('id')), 'total_sessions']
      ],
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        }
      },
      group: [fn('DATE', col('created_at'))],
      order: [[fn('DATE', col('created_at')), 'ASC']]
    });
    
    // Kết hợp dữ liệu cho endpoint /daily
    const combinedStats = dailyStats.map(stat => {
      const sessionStat = dailySessions.find(s => s.getDataValue('date') === stat.getDataValue('date'));
      return {
        date: stat.getDataValue('date'),
        total_messages: parseInt(stat.getDataValue('total_messages')),
        api_calls: Math.floor(apiCalls / days), // Phân bổ đều cho các ngày
        qa_responses: Math.floor(qaResponses / days),
        chatgpt_responses: Math.floor(chatgptResponses / days),
        errors: Math.floor(errors / days),
        total_sessions: sessionStat ? parseInt(sessionStat.getDataValue('total_sessions')) : 0
      };
    });
    
    res.json({
      success: true,
      data: {
        stats: combinedStats,
        period: `${days} days`,
        summary: {
          total_api_calls: apiCalls,
          total_qa_responses: qaResponses,
          total_chatgpt_responses: chatgptResponses,
          total_errors: errors
        }
      }
    });
  } catch (error) {
    console.error('❌ Lỗi lấy thống kê theo ngày:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Lấy thống kê theo ngày với tham số
router.get('/daily/:days', async (req, res) => {
  try {
    const days = parseInt(req.params.days) || 7;
    const { ChatSession, ChatMessage } = require('../models');
    const { Op, fn, col } = require('sequelize');
    
    // Tính ngày bắt đầu
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);
    
    // Lấy thống kê theo ngày - sử dụng cú pháp đơn giản
    const dailyStats = await ChatMessage.findAll({
      attributes: [
        [fn('DATE', col('created_at')), 'date'],
        [fn('COUNT', col('id')), 'total_messages']
      ],
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        }
      },
      group: [fn('DATE', col('created_at'))],
      order: [[fn('DATE', col('created_at')), 'ASC']]
    });

    // Lấy thống kê chi tiết theo loại message cho endpoint thứ hai
    const apiCalls2 = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        message_type: 'api_call'
      }
    });

    const qaResponses2 = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        message_type: 'qa_response'
      }
    });

    const chatgptResponses2 = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        message_type: 'bot'
      }
    });

    const errors2 = await ChatMessage.count({
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        },
        error: {
          [Op.ne]: null
        }
      }
    });
    
    // Lấy thống kê sessions theo ngày
    const dailySessions = await ChatSession.findAll({
      attributes: [
        [fn('DATE', col('created_at')), 'date'],
        [fn('COUNT', col('id')), 'total_sessions']
      ],
      where: {
        created_at: {
          [Op.between]: [startDate, endDate]
        }
      },
      group: [fn('DATE', col('created_at'))],
      order: [[fn('DATE', col('created_at')), 'ASC']]
    });
    
    // Kết hợp dữ liệu cho endpoint /daily/:days
    const combinedStats = dailyStats.map(stat => {
      const sessionStat = dailySessions.find(s => s.getDataValue('date') === stat.getDataValue('date'));
      return {
        date: stat.getDataValue('date'),
        total_messages: parseInt(stat.getDataValue('total_messages')),
        api_calls: Math.floor(apiCalls2 / days), // Phân bổ đều cho các ngày
        qa_responses: Math.floor(qaResponses2 / days),
        chatgpt_responses: Math.floor(chatgptResponses2 / days),
        errors: Math.floor(errors2 / days),
        total_sessions: sessionStat ? parseInt(sessionStat.getDataValue('total_sessions')) : 0
      };
    });
    
    res.json({
      success: true,
      data: {
        stats: combinedStats,
        period: `${days} days`,
        summary: {
          total_api_calls: apiCalls2,
          total_qa_responses: qaResponses2,
          total_chatgpt_responses: chatgptResponses2,
          total_errors: errors2
        }
      }
    });
  } catch (error) {
    console.error('❌ Lỗi lấy thống kê theo ngày:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Lấy danh sách API mappings
router.get('/api-mappings', async (req, res) => {
  try {
    const mappings = await ApiMappingService.getAll();
    res.json({
      success: true,
      data: mappings
    });
  } catch (error) {
    console.error('❌ Lỗi lấy danh sách API mappings:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Lấy chi tiết API mapping
router.get('/api-mappings/:id', async (req, res) => {
  try {
    const { id } = req.params;
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({
        success: false,
        error: 'ID không hợp lệ'
      });
    }

    const mapping = await ApiMappingService.getById(parseInt(id));
    if (!mapping) {
      return res.status(404).json({
        success: false,
        error: 'Không tìm thấy API mapping'
      });
    }

    res.json({
      success: true,
      data: mapping
    });
  } catch (error) {
    console.error('❌ Lỗi lấy chi tiết API mapping:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Tạo API mapping mới
router.post('/api-mappings', async (req, res) => {
  try {
    const mappingData = req.body;
    if (!mappingData || !mappingData.name || !mappingData.endpoint) {
      return res.status(400).json({
        success: false,
        error: 'Dữ liệu không hợp lệ'
      });
    }

    const newMapping = await ApiMappingService.create(mappingData);
    res.status(201).json({
      success: true,
      data: newMapping
    });
  } catch (error) {
    console.error('❌ Lỗi tạo API mapping:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Cập nhật API mapping
router.put('/api-mappings/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const mappingData = req.body;
    
    console.log('Raw request body:', req.body);
    console.log('Content-Type:', req.headers['content-type']);
    console.log('Request method:', req.method);

    // Validate input
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({
        success: false,
        error: 'ID không hợp lệ'
      });
    }

    if (!mappingData || typeof mappingData !== 'object') {
      return res.status(400).json({
        success: false,
        error: 'Dữ liệu cập nhật không hợp lệ'
      });
    }

    // Validate required fields if they are being updated
    if (mappingData.domain !== undefined && (!mappingData.domain || mappingData.domain.trim() === '')) {
      return res.status(400).json({
        success: false,
        error: 'Domain không được để trống'
      });
    }

    if (mappingData.api_server !== undefined && (!mappingData.api_server || mappingData.api_server.trim() === '')) {
      return res.status(400).json({
        success: false,
        error: 'API Server không được để trống'
      });
    }

    console.log(`🔄 Cập nhật API mapping ID: ${id}`, mappingData);

    const updatedMapping = await ApiMappingService.update(parseInt(id), mappingData);
    
    if (!updatedMapping) {
      return res.status(404).json({
        success: false,
        error: 'Không tìm thấy API mapping để cập nhật'
      });
    }

    res.json({
      success: true,
      data: updatedMapping,
      message: 'Cập nhật thành công'
    });
  } catch (error) {
    console.error('❌ Lỗi cập nhật API mapping:', error);
    
    // Xử lý các loại lỗi cụ thể
    if (error.name === 'SequelizeValidationError') {
      return res.status(400).json({
        success: false,
        error: 'Dữ liệu không hợp lệ: ' + error.message
      });
    }
    
    if (error.name === 'SequelizeUniqueConstraintError') {
      return res.status(400).json({
        success: false,
        error: 'Dữ liệu đã tồn tại trong hệ thống'
      });
    }
    
    res.status(500).json({
      success: false,
      error: 'Lỗi server: ' + error.message
    });
  }
});

// Xóa API mapping
router.delete('/api-mappings/:id', async (req, res) => {
  try {
    const { id } = req.params;
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({
        success: false,
        error: 'ID không hợp lệ'
      });
    }

    const deleted = await ApiMappingService.delete(parseInt(id));
    if (!deleted) {
      return res.status(404).json({
        success: false,
        error: 'Không tìm thấy API mapping để xóa'
      });
    }

    res.json({
      success: true,
      message: 'Xóa thành công'
    });
  } catch (error) {
    console.error('❌ Lỗi xóa API mapping:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Lấy danh sách Q&A Database
router.get('/qa-database', async (req, res) => {
  try {
    const qaData = await QADatabaseService.getAll();
    res.json({
      success: true,
      data: qaData
    });
  } catch (error) {
    console.error('❌ Lỗi lấy Q&A Database:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Lấy chi tiết Q&A Database
router.get('/qa-database/:id', async (req, res) => {
  try {
    const { id } = req.params;
    if (!id || isNaN(parseInt(id))) {
      return res.status(400).json({
        success: false,
        error: 'ID không hợp lệ'
      });
    }

    const qaItem = await QADatabaseService.getById(parseInt(id));
    if (!qaItem) {
      return res.status(404).json({
        success: false,
        error: 'Không tìm thấy Q&A item'
      });
    }

    res.json({
      success: true,
      data: qaItem
    });
  } catch (error) {
    console.error('❌ Lỗi lấy chi tiết Q&A Database:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Tạo Q&A Database mới
router.post('/qa-database', async (req, res) => {
  try {
    const qaData = req.body;
    if (!qaData || !qaData.question || !qaData.answer) {
      return res.status(400).json({
        success: false,
        error: 'Dữ liệu không hợp lệ'
      });
    }

    const newQA = await QADatabaseService.create(qaData);
    res.status(201).json({
      success: true,
      data: newQA
    });
  } catch (error) {
    console.error('❌ Lỗi tạo Q&A Database:', error);
    res.status(500).json({
      success: false,
      error: 'Lỗi server'
    });
  }
});

// Test endpoint để kiểm tra việc nhận dữ liệu
router.post('/test-mapping', async (req, res) => {
  try {
    console.log('Test endpoint - Request body:', req.body);
    console.log('Test endpoint - Request headers:', req.headers);
    console.log('Test endpoint - Content-Type:', req.headers['content-type']);
    
    res.json({
      success: true,
      data: {
        receivedBody: req.body,
        contentType: req.headers['content-type'],
        method: req.method
      }
    });
  } catch (error) {
    console.error('❌ Test endpoint error:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

module.exports = router;
