const { Sequelize } = require('sequelize');

// Cấu hình database
const sequelize = new Sequelize({
  dialect: 'mysql',
  host: process.env.DB_HOST || 'localhost',
  port: process.env.DB_PORT || 3306,
  username: process.env.DB_USER || 'root',
  password: process.env.DB_PASSWORD || '',
  database: process.env.DB_NAME || 'chatbot_gpt',
  logging: process.env.NODE_ENV === 'development' ? console.log : false,
  pool: {
    max: 20,
    min: 5,
    acquire: 60000,
    idle: 30000
  },
  define: {
    timestamps: true,
    underscored: true,
    freezeTableName: true
  },
  dialectOptions: {
    charset: 'utf8mb4',
    supportBigNumbers: true,
    bigNumberStrings: true,
    connectTimeout: 60000,
    acquireTimeout: 60000,
    timeout: 60000
  },
  retry: {
    max: 3,
    timeout: 3000
  }
});

// Test kết nối
async function testConnection() {
  try {
    await sequelize.authenticate();
    console.log('✅ Kết nối database thành công!');
    return true;
  } catch (error) {
    console.error('❌ Lỗi kết nối database:', error.message);
    return false;
  }
}

// Sync database (tạo bảng nếu chưa có)
async function syncDatabase() {
  try {
    await sequelize.sync({ alter: true });
    console.log('✅ Database đã được đồng bộ!');
    return true;
  } catch (error) {
    console.error('❌ Lỗi đồng bộ database:', error.message);
    return false;
  }
}

// Kiểm tra kết nối và tạo pool
async function ensureConnection() {
  try {
    // Kiểm tra xem connection pool có đang hoạt động không
    if (sequelize.connectionManager.pool) {
      const pool = sequelize.connectionManager.pool;
      if (pool.destroyed) {
        console.log('🔄 Reconnecting to database...');
        await sequelize.authenticate();
      }
    } else {
      await sequelize.authenticate();
    }
    return true;
  } catch (error) {
    console.error('❌ Lỗi kết nối database:', error.message);
    return false;
  }
}

module.exports = {
  sequelize,
  testConnection,
  syncDatabase,
  ensureConnection
};
