-- ========================================
-- CHATBOT GPT DATABASE SETUP
-- ========================================

-- Tạo database
CREATE DATABASE IF NOT EXISTS `chatbot_gpt` 
CHARACTER SET utf8mb4 
COLLATE utf8mb4_unicode_ci;

-- Sử dụng database
USE `chatbot_gpt`;

-- Tạo user cho ứng dụng (tùy chọn - cho local development)
-- CREATE USER IF NOT EXISTS 'chatbot_user'@'localhost' IDENTIFIED BY 'your_secure_password';
-- GRANT ALL PRIVILEGES ON chatbot_gpt.* TO 'chatbot_user'@'localhost';
-- FLUSH PRIVILEGES;

-- Tạo bảng API Mappings
CREATE TABLE IF NOT EXISTS `api_mappings` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `domain` varchar(255) NOT NULL,
  `api_server` varchar(500) NOT NULL,
  `environment` varchar(50) NOT NULL DEFAULT 'development',
  `description` text,
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `domain` (`domain`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tạo bảng Q&A Database
CREATE TABLE IF NOT EXISTS `qa_database` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `question` text NOT NULL,
  `answer` text NOT NULL,
  `category` varchar(100) DEFAULT 'general',
  `keywords` text,
  `score` int(11) NOT NULL DEFAULT 0,
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_category` (`category`),
  KEY `idx_score` (`score`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tạo bảng Chat Sessions
CREATE TABLE IF NOT EXISTS `chat_sessions` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `session_id` varchar(255) NOT NULL,
  `thread_id` varchar(255) DEFAULT NULL,
  `user_info` json DEFAULT NULL,
  `api_mapping` json DEFAULT NULL,
  `last_activity` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `session_id` (`session_id`),
  KEY `idx_last_activity` (`last_activity`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tạo bảng Chat Messages
CREATE TABLE IF NOT EXISTS `chat_messages` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `session_id` varchar(255) NOT NULL,
  `message_type` enum('user','bot','api_call','qa_response') NOT NULL,
  `content` text NOT NULL,
  `response` text,
  `api_type` varchar(50) DEFAULT NULL,
  `api_params` json DEFAULT NULL,
  `api_result` json DEFAULT NULL,
  `processing_time` int(11) DEFAULT NULL COMMENT 'milliseconds',
  `domain` varchar(255) DEFAULT NULL,
  `user_agent` text,
  `ip_address` varchar(45) DEFAULT NULL,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_session_id` (`session_id`),
  KEY `idx_created_at` (`created_at`),
  KEY `idx_message_type` (`message_type`),
  KEY `fk_session_id` (`session_id`),
  CONSTRAINT `fk_session_id` FOREIGN KEY (`session_id`) REFERENCES `chat_sessions` (`session_id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tạo bảng Chat Statistics
CREATE TABLE IF NOT EXISTS `chat_statistics` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `date` date NOT NULL,
  `total_sessions` int(11) NOT NULL DEFAULT 0,
  `total_messages` int(11) NOT NULL DEFAULT 0,
  `api_calls` int(11) NOT NULL DEFAULT 0,
  `qa_responses` int(11) NOT NULL DEFAULT 0,
  `gpt_responses` int(11) NOT NULL DEFAULT 0,
  `avg_response_time` float NOT NULL DEFAULT 0,
  `unique_users` int(11) NOT NULL DEFAULT 0,
  `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `date` (`date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Thêm dữ liệu mẫu cho API Mappings
INSERT INTO `api_mappings` (`domain`, `api_server`, `environment`, `description`) VALUES
('localhost', 'https://kthcdemoapi.dtsoft.vn/api', 'development', 'Local Development'),
('localhost:3000', 'https://kthcdemoapi.dtsoft.vn/api', 'development', 'Local Development Port 3000'),
('127.0.0.1:3000', 'https://kthcdemoapi.dtsoft.vn/api', 'development', 'Local Development IP'),
('kthcdemo.dtsoft.vn', 'https://kthcdemoapi.dtsoft.vn/api', 'production', 'KTHC Demo Production'),
('kthctravinh.dtsoft.vn', 'https://apikthctravinh.dtsoft.vn/api', 'production', 'KTHC Trà Vinh Production'),
('kthccantho.dtsoft.vn', 'https://apikthccantho.dtsoft.vn/api', 'production', 'KTHC Cần Thơ Production'),
('kthchcm.dtsoft.vn', 'https://apikthchcm.dtsoft.vn/api', 'production', 'KTHC HCM Production'),
('kthcdanang.dtsoft.vn', 'https://apikthcdanang.dtsoft.vn/api', 'production', 'KTHC Đà Nẵng Production')
ON DUPLICATE KEY UPDATE 
  `api_server` = VALUES(`api_server`),
  `environment` = VALUES(`environment`),
  `description` = VALUES(`description`),
  `updated_at` = CURRENT_TIMESTAMP;

-- Thêm dữ liệu mẫu cho Q&A Database
INSERT INTO `qa_database` (`question`, `answer`, `category`, `keywords`, `score`) VALUES
('KTHC Online là gì?', 'KTHC Online là hệ thống quản lý tài chính kế toán trực tuyến, giúp doanh nghiệp quản lý lương, bảo hiểm, công đoàn và các nghiệp vụ tài chính khác một cách hiệu quả.', 'general', 'KTHC,Online,hệ thống,quản lý,tài chính,kế toán', 10),
('Cách tạo lương tháng?', 'Để tạo lương tháng, bạn cần: 1) Vào menu Lương > Tạo lương tháng, 2) Chọn tháng cần tạo, 3) Nhập thông tin nhân viên, 4) Tính toán và lưu.', 'luong', 'tạo,lương,tháng,salary,payroll', 8),
('Cách tạo bảo hiểm tháng?', 'Để tạo bảo hiểm tháng: 1) Vào menu Bảo hiểm > Tạo bảo hiểm tháng, 2) Chọn tháng, 3) Nhập thông tin nhân viên, 4) Tính toán BHXH, BHYT, BHTN.', 'baohiem', 'tạo,bảo hiểm,tháng,insurance,BHXH,BHYT,BHTN', 8),
('Cách tạo kinh phí công đoàn?', 'Để tạo kinh phí công đoàn: 1) Vào menu Công đoàn > Tạo kinh phí công đoàn, 2) Chọn tháng, 3) Tính 2% lương cơ bản, 4) Lưu và in báo cáo.', 'congdoan', 'tạo,kinh phí,công đoàn,union,KPCD', 8)
ON DUPLICATE KEY UPDATE 
  `answer` = VALUES(`answer`),
  `category` = VALUES(`category`),
  `keywords` = VALUES(`keywords`),
  `score` = VALUES(`score`),
  `updated_at` = CURRENT_TIMESTAMP;

-- Tạo indexes cho performance
CREATE INDEX `idx_api_mappings_active` ON `api_mappings` (`is_active`);
CREATE INDEX `idx_qa_database_active` ON `qa_database` (`is_active`);
CREATE INDEX `idx_chat_sessions_active` ON `chat_sessions` (`is_active`);
CREATE INDEX `idx_chat_statistics_date` ON `chat_statistics` (`date`);

-- Hiển thị thông tin database
SELECT 'Database chatbot_gpt đã được tạo thành công!' AS message;
SELECT COUNT(*) AS total_api_mappings FROM api_mappings;
SELECT COUNT(*) AS total_qa_records FROM qa_database;
