/**
 * 🤖 ChatBot GPT Widget
 * File JavaScript độc lập để nhúng ChatBot vào website khác
 * Sử dụng: Chỉ cần include file này và cấu hình
 */

(function() {
    'use strict';

    // Server Configuration - LUÔN CỐ ĐỊNH
    const SERVER_CONFIG = {
        // Server URL chính - LUÔN CỐ ĐỊNH
        BASE_URL: 'https://api.muvietplus.net',
        
        // Các endpoint API
        ENDPOINTS: {
            CHAT: '/api/chat',
            CTGS_CREATE: '/api/ctgs/create',
            CTGS_LIST: '/api/ctgs/list',
            LUONG: '/api/luong',
            BAOHIEM: '/api/baohiem',
            CONGDOAN: '/api/congdoan',
            MAPPING_INFO: '/api/mapping-info',
            DOMAINS: '/api/domains'
        }
    };

    // Dynamic API URL based on current domain
    function getApiBaseUrl() {
        const currentDomain = window.location.hostname;
        const currentPort = window.location.port;
        
        // Nếu đang ở localhost, sử dụng localhost
        if (currentDomain === 'localhost' || currentDomain === '127.0.0.1') {
            const protocol = window.location.protocol;
            const port = currentPort ? `:${currentPort}` : '';
            return `${protocol}//${currentDomain}${port}`;
        }
        
        // Nếu không phải localhost, sử dụng server cố định
        return SERVER_CONFIG.BASE_URL;
    }

    // Helper function để lấy API URL động
    function getApiUrl(endpoint) {
        const baseUrl = getApiBaseUrl();
        return baseUrl + (endpoint.startsWith('/') ? endpoint : '/' + endpoint);
    }

    // API Mapping Info Function - Chỉ để log thông tin
    async function getApiMappingInfo() {
        try {
            console.log('🗺️ Lấy thông tin API Mapping...');
            
            // Lấy domain hiện tại
            const currentDomain = window.location.hostname;
            console.log('🌐 Current domain:', currentDomain);
            
            // Gọi API để lấy mapping info (chỉ để log)
            const response = await fetch(getApiUrl(SERVER_CONFIG.ENDPOINTS.MAPPING_INFO), {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'Origin': window.location.origin
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success && data.data) {
                    console.log('✅ API Mapping Info:', {
                        domain: data.data.current_domain,
                        api_server: data.data.api_server,
                        environment: data.data.environment,
                        description: data.data.description
                    });
                    return data.data;
                }
            }
            
            return null;
            
        } catch (error) {
            console.error('❌ Lỗi khi lấy API mapping info:', error);
            return null;
        }
    }

    // Cấu hình mặc định
    const DEFAULT_CONFIG = {
        // API Configuration
        apiUrl: getApiUrl(SERVER_CONFIG.ENDPOINTS.CHAT),
        sessionId: null,
        
        // UI Configuration
        showPopupIcon: true,
        autoOpen: false,
        position: 'bottom-right', // bottom-right, bottom-left, top-right, top-left
        
        // Theme Configuration
        theme: {
            primaryColor: '#667eea',
            secondaryColor: '#764ba2',
            backgroundColor: '#ffffff',
            textColor: '#333333'
        },
        
        // Behavior Configuration
        autoFocus: true,
        showNotification: true,
        enableSound: false,
        
        // Customization
        title: 'DTSBot',
        welcomeMessage: 'Chào mừng bạn! Tôi là trợ lý DTSBot, bạn cần hỗ trợ gì.',
        placeholder: 'Nhập tin nhắn của bạn...',
        
        // Callbacks
        onOpen: null,
        onClose: null,
        onMessage: null,
        onError: null
    };

    // Global configuration
    window.CHATBOT_CONFIG = Object.assign({}, DEFAULT_CONFIG);

    // Cho phép website chủ thay đổi cấu hình
    window.setChatbotConfig = function(config) {
        Object.assign(window.CHATBOT_CONFIG, config);
    };

    // Function để thay đổi server URL
    window.setServerUrl = function(baseUrl) {
        SERVER_CONFIG.BASE_URL = baseUrl;
        // Cập nhật lại apiUrl trong config
        window.CHATBOT_CONFIG.apiUrl = SERVER_CONFIG.BASE_URL + SERVER_CONFIG.ENDPOINTS.CHAT;
        console.log('✅ Server URL đã thay đổi thành:', baseUrl);
        console.log('🔗 API URL mới:', window.CHATBOT_CONFIG.apiUrl);
    };

    // Function để lấy server URL hiện tại
    window.getServerUrl = function() {
        return SERVER_CONFIG.BASE_URL;
    };

    // Function để lấy endpoint URL đầy đủ
    window.getApiUrl = function(endpoint) {
        return SERVER_CONFIG.BASE_URL + (SERVER_CONFIG.ENDPOINTS[endpoint] || endpoint);
    };

    // Debug functions
    window.chatbotDebug = {
        clearAll: function() {
            if (window.ChatBotAPI && window.ChatBotAPI.debugClearAll) {
                window.ChatBotAPI.debugClearAll();
            } else {
                console.log('🧹 Debug: Xóa tất cả localStorage liên quan đến ChatBot');
                localStorage.removeItem('chatbot_session_id');
                localStorage.removeItem('chatbot_widget_state');
                console.log(' Đã xóa tất cả localStorage');
                console.log('🔍 localStorage hiện tại:', Object.keys(localStorage));
            }
        },
        reset: function() {
            if (window.ChatBotAPI && window.ChatBotAPI.resetWidget) {
                window.ChatBotAPI.resetWidget();
            } else {
                console.log('❌ ChatBotAPI chưa sẵn sàng');
            }
        }
    };

    // CTGS Functions
    window.createCTGS = function() {
        console.log('🔍 createCTGS được gọi');
        
        try {
            const form = document.getElementById('ctgsForm');
            
            // Kiểm tra form có tồn tại không
            if (!form) {
                console.error('❌ Không tìm thấy form #ctgsForm');
                alert('❌ Lỗi: Không tìm thấy form');
                return;
            }
            
            // Kiểm tra validation
            const shctgs = form.querySelector('input[name="Shctgs"]').value.trim();
            if (!shctgs) {
                alert('❌ Vui lòng nhập Số hiệu');
                form.querySelector('input[name="Shctgs"]').focus();
                return;
            }
            
            const formData = new FormData(form);
            const ctgsData = {};
            
            for (let [key, value] of formData.entries()) {
                ctgsData[key] = value;
            }
          
            // Sử dụng giá trị từ localStorage thay vì userInfo
            ctgsData.MaCoQuan = parseInt(localStorage.getItem('kthcMaCoQuan')) || 73;
            ctgsData.MaCanBo = parseInt(localStorage.getItem('kthcMaCanBo')) || 104;
            ctgsData.Nam = parseInt(localStorage.getItem('kthcNam')) || 2023;
            
            console.log('CTGS Data:', ctgsData);
            
            // Lấy API params từ localStorage
            const apiParams = {
                user: localStorage.getItem('user') || '',
                token_httc: localStorage.getItem('token_httc') || '',
                kthcMaCoQuan: parseInt(localStorage.getItem('kthcMaCoQuan')) || 73,
                kthcMaCanBo: parseInt(localStorage.getItem('kthcMaCanBo')) || 104,
                kthcNam: parseInt(localStorage.getItem('kthcNam')) || 2023,
                kthcThang: parseInt(localStorage.getItem('kthcThang')) || 12,
                kthcplgd: localStorage.getItem('kthcplgd') || 'KH-CB',
                kthcIdvunglv: parseInt(localStorage.getItem('kthcIdvunglv')) || 68,
                kthcNlb: localStorage.getItem('kthcNlb') || '',
                kthcPtkt: localStorage.getItem('kthcPtkt') || ''
            };

            console.log('🔍 Debug - API Params từ localStorage:', apiParams);
            console.log('🔍 Debug - Request payload:', {
                ctgsData: ctgsData,
                ...apiParams
            });

            fetch(getApiUrl('/api/ctgs/create'), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    ctgsData: ctgsData,
                    ...apiParams
                })
            })
            .then(response => {
                console.log('🔍 Debug - Response status:', response.status);
                return response.json();
            })
                        .then(data => {
              console.log('🔍 Debug - Response data:', data);
              
              // Kiểm tra cả success và trực tiếp Idctgs
              if (data.success || data.Idctgs) {
                const idctgs = data.Idctgs || (data.data && data.data.idctgs);
                const soHieu = document.querySelector('input[name="Shctgs"]').value;
                
                // Thay thế form bằng thông báo thành công
                const formContainer = document.querySelector('#ctgsForm').closest('div');
                if (formContainer) {
                  formContainer.innerHTML = `
                    <div style="padding: 20px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 5px; color: #155724;">
                      <h4 style="margin: 0 0 10px 0; color: #155724;">✅ Đã tạo CTGS thành công!</h4>
                      <p style="margin: 0 0 10px 0;"><strong>Số hiệu:</strong> ${soHieu}</p>
                      <p style="margin: 0 0 15px 0;"><strong>ID:</strong> ${idctgs}</p>
                      <p style="margin: 0; font-style: italic; color: #0f5132;">🔄 Đang thao tác lại nghiệp vụ bạn muốn hạch toán...</p>
                    </div>
                  `;
                }
                
                // Tự động gọi lại lệnh từ context hiện tại sau 2 giây
                setTimeout(() => {
                  // Lấy context từ localStorage hoặc mặc định là "Tạo lương tháng"
                  const currentContext = localStorage.getItem('currentContext') || 'Tạo lương tháng';
                  console.log('🔍 Tự động gọi lại lệnh từ context:', currentContext);
                  
                  if (window.chatbotWidget && window.chatbotWidget.guiTinNhanCoCo) {
                    window.chatbotWidget.guiTinNhanCoCo(currentContext, true);
                  } else if (window.ChatBotAPI && window.ChatBotAPI.sendMessage) {
                    window.ChatBotAPI.sendMessage(currentContext);
                  } else {
                    console.log('⚠️ Không tìm thấy ChatBotAPI, reload trang...');
                    window.location.reload();
                  }
                }, 2000);
                
              } else {
                alert('❌ Lỗi: ' + (data.error || 'Không thể tạo CTGS'));
              }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('❌ Lỗi kết nối: ' + error.message);
            });
        } catch (error) {
            console.error('❌ Lỗi trong createCTGS:', error);
            alert('❌ Lỗi xử lý form: ' + error.message);
        }
    };

    window.cancelCTGS = function() {
        console.log('🔍 cancelCTGS được gọi');
        if (confirm('Bạn có chắc muốn hủy tạo CTGS?')) {
            const formContainer = document.querySelector('#ctgsForm').closest('div');
            if (formContainer) {
                formContainer.remove();
            }
        }
    };

    // CSS Styles
    const CSS_STYLES = `
        .chatbot-widget * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        .chatbot-widget {
            font-family: Arial, sans-serif;
            position: fixed;
            z-index: 999999;
        }

        /* Popup Icon */
        .chatbot-popup-icon {
            position: fixed;
            bottom: 20px;
            right: 20px;
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, var(--primary-color, #667eea) 0%, var(--secondary-color, #764ba2) 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            transition: all 0.3s ease;
            z-index: 999997;
        }

        .chatbot-popup-icon:hover {
            transform: scale(1.1);
            box-shadow: 0 6px 20px rgba(0,0,0,0.3);
        }

        .chatbot-popup-icon i {
            color: white;
            font-size: 24px;
        }

        .chatbot-notification-badge {
            position: absolute;
            top: -5px;
            right: -5px;
            background: #ff4757;
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            font-weight: bold;
        }

        /* Chat Container */
        .chatbot-overlay {
            position: fixed;
            bottom: 90px;
            right: 20px;
            width: 450px;
            height: 600px;
            background: transparent;
            display: none;
            z-index: 999998;
            pointer-events: auto;
        }

        .chatbot-container {
            width: 100%;
            height: 100%;
            background: var(--background-color, #ffffff);
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        /* Header */
        .chatbot-header {
            background: linear-gradient(135deg, var(--primary-color, #667eea) 0%, var(--secondary-color, #764ba2) 100%);
            color: white;
            padding: 0px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            min-height: 60px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .chatbot-logo {
            display: flex;
            align-items: center;
            gap: 12px;
            flex: 1;
        }

        .chatbot-logo i {
            font-size: 26px;
            color: white;
            opacity: 0.9;
        }

        .chatbot-logo h1 {
            font-size: 19px;
            font-weight: 600;
            color: white;
            margin: 0;
            letter-spacing: 0.5px;
        }

        .chatbot-session-info {
            display: flex;
            align-items: center;
            gap: 8px;
            flex-shrink: 0;
        }

        .chatbot-btn {
            background: rgba(255,255,255,0.15);
            border: 1px solid rgba(255,255,255,0.2);
            color: white;
            padding: 6px 10px;
            border-radius: 16px;
            cursor: pointer;
            font-size: 11px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 4px;
            transition: all 0.3s ease;
            white-space: nowrap;
        }

        .chatbot-btn:hover {
            background: rgba(255,255,255,0.25);
            border-color: rgba(255,255,255,0.3);
            transform: translateY(-1px);
        }

        .chatbot-btn i {
            font-size: 10px;
        }

        .chatbot-reset-btn {
            background: rgba(255, 193, 7, 0.2);
            border-color: rgba(255, 193, 7, 0.3);
        }

        .chatbot-reset-btn:hover {
            background: rgba(255, 193, 7, 0.3);
            border-color: rgba(255, 193, 7, 0.4);
        }

        .chatbot-btn-close {
            background: rgba(255,255,255,0.15);
            border: 1px solid rgba(255,255,255,0.2);
            color: white;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }

        .chatbot-btn-close:hover {
            background: rgba(255,255,255,0.25);
            border-color: rgba(255,255,255,0.3);
            transform: translateY(-1px);
        }

        .chatbot-btn-close i {
            font-size: 12px;
        }

        /* Body */
        .chatbot-body {
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
            position: relative;
        }

        .chatbot-connection-status {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 40px 30px;
            text-align: center;
            color: #666;
            min-width: 250px;
            z-index: 10;
        }

        .chatbot-connection-status-icon {
            font-size: 64px;
            color: var(--primary-color, #667eea);
            margin-bottom: 20px;
            animation: chatbotPulse 2s infinite;
        }

        @keyframes chatbotPulse {
            0% { opacity: 0.6; transform: scale(1); }
            50% { opacity: 1; transform: scale(1.05); }
            100% { opacity: 0.6; transform: scale(1); }
        }

        .chatbot-connection-status-text {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
        }

        .chatbot-connection-status-subtitle {
            font-size: 14px;
            color: #666;
            line-height: 1.4;
        }

        .chatbot-messages {
            flex: 1;
            overflow-y: auto;
            padding: 20px;
            display: flex;
            flex-direction: column;
            gap: 15px;
        }

        .chatbot-welcome-message {
            text-align: center;
            color: #666;
            padding: 20px;
        }

        .chatbot-welcome-icon {
            font-size: 48px;
            color: #ddd;
            margin-bottom: 15px;
        }

        .chatbot-welcome-message h2 {
            margin: 0 0 10px 0;
            font-size: 18px;
            color: #333;
        }

        .chatbot-welcome-message p {
            margin: 0;
            font-size: 14px;
        }

        .chatbot-system-message {
            text-align: center;
            margin: 10px 0;
            animation: chatbotFadeIn 0.3s ease;
        }

        .chatbot-system-content {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 6px 12px;
            background: rgba(102, 126, 234, 0.1);
            border: 1px solid rgba(102, 126, 234, 0.2);
            border-radius: 12px;
            font-size: 12px;
            color: #667eea;
        }

        .chatbot-system-content i {
            font-size: 10px;
        }

        .chatbot-message {
            display: flex;
            gap: 10px;
            animation: chatbotFadeIn 0.3s ease;
        }

        @keyframes chatbotFadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .chatbot-message.user {
            flex-direction: row-reverse;
        }

        .chatbot-message-avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            flex-shrink: 0;
        }

        .chatbot-message.user .chatbot-message-avatar {
            background: linear-gradient(135deg, var(--primary-color, #667eea) 0%, var(--secondary-color, #764ba2) 100%);
            color: white;
        }

        .chatbot-message.bot .chatbot-message-avatar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .chatbot-message-content {
            max-width: 70%;
            padding: 12px 16px;
            border-radius: 18px;
            font-size: 14px;
            line-height: 1.4;
            word-wrap: break-word;
        }

        .chatbot-message.user .chatbot-message-content {
            background: linear-gradient(135deg, var(--primary-color, #667eea) 0%, var(--secondary-color, #764ba2) 100%);
            color: white;
            border-bottom-right-radius: 4px;
        }

        .chatbot-message.bot .chatbot-message-content {
            background: #f1f3f4;
            color: var(--text-color, #333);
            border-bottom-left-radius: 4px;
        }

        .chatbot-typing-indicator {
            display: none;
            align-items: center;
            gap: 10px;
            padding: 12px 16px;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 1px solid #dee2e6;
            border-radius: 18px;
            border-bottom-left-radius: 4px;
            max-width: 70%;
            animation: chatbotFadeIn 0.3s ease;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .chatbot-typing-indicator.show {
            display: flex;
        }

        .chatbot-typing-indicator span {
            font-size: 12px;
            color: #495057;
            font-weight: 500;
            white-space: nowrap;
        }

        .chatbot-typing-avatar {
            width: 28px;
            height: 28px;
            border-radius: 50%;
            background: #f1f3f4;
            color: #666;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            flex-shrink: 0;
        }

        .chatbot-typing-dots {
            display: flex;
            gap: 3px;
        }

        .chatbot-typing-dot {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color, #667eea) 0%, var(--secondary-color, #764ba2) 100%);
            animation: chatbotTyping 1.4s infinite ease-in-out;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }

        .chatbot-typing-dot:nth-child(1) { animation-delay: -0.32s; }
        .chatbot-typing-dot:nth-child(2) { animation-delay: -0.16s; }

        @keyframes chatbotTyping {
            0%, 80%, 100% {
                transform: scale(0.7);
                opacity: 0.4;
            }
            40% {
                transform: scale(1.1);
                opacity: 1;
            }
        }

        /* Input */
        .chatbot-input-container {
            padding: 20px;
            border-top: 1px solid #e9ecef;
            flex-shrink: 0;
            background: white;
        }

        .chatbot-input-wrapper {
            display: flex;
            align-items: flex-end;
            gap: 12px;
            background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
            border: 2px solid #e9ecef;
            border-radius: 28px;
            padding: 10px 16px;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .chatbot-input-wrapper:focus-within {
            border-color: var(--primary-color, #667eea);
            box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1), 0 4px 16px rgba(0,0,0,0.1);
            transform: translateY(-1px);
        }

        .chatbot-message-input {
            flex: 1;
            border: none;
            outline: none;
            background: transparent;
            font-size: 14px;
            line-height: 2.5;
            max-height: 120px;
            min-height: 24px;
            resize: none;
            font-family: inherit;
            color: #495057;
        }

        .chatbot-message-input::placeholder {
            color: #adb5bd;
            font-style: italic;
        }

        .chatbot-send-btn {
            width: 40px;
            height: 40px;
            border: none;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color, #667eea) 0%, var(--secondary-color, #764ba2) 100%);
            color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            flex-shrink: 0;
            box-shadow: 0 2px 8px rgba(102, 126, 234, 0.2);
        }

        .chatbot-send-btn:hover:not(:disabled) {
            transform: scale(1.08);
            box-shadow: 0 6px 16px rgba(102, 126, 234, 0.4);
        }

        .chatbot-send-btn:active:not(:disabled) {
            transform: scale(0.95);
        }

        .chatbot-send-btn:disabled {
            background: linear-gradient(135deg, #adb5bd 0%, #6c757d 100%);
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }

        .chatbot-send-btn i {
            font-size: 14px;
            transition: transform 0.2s ease;
        }

        .chatbot-send-btn:hover:not(:disabled) i {
            transform: translateX(1px);
        }

        .chatbot-suggestions-btn {
            width: 36px;
            height: 36px;
            border: none;
            border-radius: 50%;
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            flex-shrink: 0;
            box-shadow: 0 2px 8px rgba(40, 167, 69, 0.2);
            position: relative;
        }

        .chatbot-suggestions-btn:hover:not(:disabled) {
            transform: scale(1.08);
            box-shadow: 0 6px 16px rgba(40, 167, 69, 0.4);
        }

        .chatbot-suggestions-btn:active:not(:disabled) {
            transform: scale(0.95);
        }

        .chatbot-suggestions-btn:disabled {
            background: linear-gradient(135deg, #adb5bd 0%, #6c757d 100%);
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }

        .chatbot-suggestions-btn i {
            font-size: 14px;
            transition: transform 0.2s ease;
        }

        .chatbot-suggestions-btn:hover:not(:disabled) i {
            transform: rotate(45deg);
        }

        .chatbot-input-container {
            padding: 20px;
            border-top: 1px solid #e9ecef;
            flex-shrink: 0;
            background: white;
            position: relative;
        }

        .chatbot-suggestions-menu {
            position: absolute;
            bottom: 100%;
            left: 20px;
            right: 20px;
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0,0,0,0.1);
            margin-bottom: 10px;
            opacity: 0;
            visibility: hidden;
            transform: translateY(10px);
            transition: all 0.3s ease;
            z-index: 1000;
        }

        .chatbot-suggestions-menu.show {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }

        .chatbot-suggestions-header {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 12px 16px;
            border-bottom: 1px solid #e9ecef;
            color: #495057;
            font-weight: 600;
            font-size: 14px;
        }

        .chatbot-suggestions-header i {
            color: #ffc107;
        }

        .chatbot-suggestions-list {
            padding: 8px;
        }

        .chatbot-suggestion-item {
            display: flex;
            align-items: center;
            gap: 10px;
            width: 100%;
            padding: 10px 12px;
            border: none;
            background: transparent;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s ease;
            color: #495057;
            font-size: 13px;
            text-align: left;
        }

        .chatbot-suggestion-item:hover {
            background: #f8f9fa;
            transform: translateX(2px);
        }

        .chatbot-suggestion-item i {
            width: 16px;
            color: #667eea;
            font-size: 12px;
        }

        /* Suggestions Tab Styles */
        .chatbot-suggestions-tabs {
            display: flex;
            border-bottom: 1px solid #e9ecef;
        }

        .chatbot-suggestion-tab {
            flex: 1;
            padding: 10px 12px;
            border: none;
            background: transparent;
            cursor: pointer;
            font-size: 12px;
            font-weight: 500;
            color: #6c757d;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 6px;
        }

        .chatbot-suggestion-tab:hover {
            background: #f8f9fa;
            color: #667eea;
        }

        .chatbot-suggestion-tab.active {
            background: #fff;
            color: #667eea;
            border-bottom: 2px solid #667eea;
        }

        .chatbot-suggestions-content {
            position: relative;
        }

        .chatbot-suggestion-pane {
            display: none;
        }

        .chatbot-suggestion-pane.active {
            display: block;
        }

        .chatbot-api-badge {
            display: inline-flex;
            align-items: center;
            gap: 4px;
            padding: 2px 6px;
            background: rgba(102, 126, 234, 0.1);
            border: 1px solid rgba(102, 126, 234, 0.2);
            border-radius: 8px;
            font-size: 10px;
            color: #667eea;
            margin-top: 4px;
            font-weight: 500;
        }

        .chatbot-api-badge i {
            font-size: 8px;
        }

        .chatbot-message.api-call .chatbot-message-content {
            border-left: 3px solid #667eea;
        }

        /* Table Styles */
        .chatbot-table-container {
            margin: 15px 0;
            overflow-x: auto;
        }

        .chatbot-table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .chatbot-table th {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 15px;
            text-align: left;
            font-weight: 600;
            font-size: 13px;
        }

        .chatbot-table td {
            padding: 10px 15px;
            border-bottom: 1px solid #e9ecef;
            font-size: 13px;
        }

        .chatbot-table tr:hover {
            background: #f8f9fa;
        }

        .chatbot-table tr:last-child td {
            border-bottom: none;
        }

        /* Ranking Styles */
        .chatbot-ranking {
            margin: 15px 0;
        }

        .chatbot-rank-item {
            display: flex;
            align-items: center;
            padding: 8px 12px;
            margin: 4px 0;
            background: #f8f9fa;
            border-radius: 6px;
            border-left: 3px solid #667eea;
        }

        .chatbot-rank-number {
            background: #667eea;
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 11px;
            font-weight: bold;
            margin-right: 12px;
        }

        .chatbot-rank-name {
            flex: 1;
            font-weight: 500;
            color: #495057;
        }

        .chatbot-rank-salary {
            font-weight: bold;
            color: #28a745;
            font-size: 12px;
        }

        /* Action Buttons */
        .chatbot-action-buttons {
            display: flex;
            gap: 8px;
            margin: 15px 0;
            flex-wrap: wrap;
        }

        .chatbot-action-btn {
            padding: 8px 16px;
            border: none;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 4px;
        }

        .chatbot-action-btn.primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .chatbot-action-btn.secondary {
            background: #6c757d;
            color: white;
        }

        .chatbot-action-btn.success {
            background: #28a745;
            color: white;
        }

        .chatbot-action-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }

        .chatbot-action-btn:active {
            transform: translateY(0);
        }

        /* CTGS Card Styles */
        .chatbot-ctgs-list {
            margin-top: 10px;
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .chatbot-ctgs-card {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 12px;
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            transition: all 0.2s ease;
        }
        
        .chatbot-ctgs-card:hover {
            border-color: var(--primary-color);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
            transform: translateY(-1px);
        }
        
        .chatbot-ctgs-info {
            display: flex;
            align-items: center;
            gap: 12px;
            flex: 1;
        }
        
        .chatbot-ctgs-number {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 24px;
            height: 24px;
            background: var(--primary-color);
            color: white;
            border-radius: 50%;
            font-size: 12px;
            font-weight: 600;
        }
        
        .chatbot-ctgs-details {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }
        
        .chatbot-ctgs-title {
            font-size: 13px;
            font-weight: 600;
            color: #333;
        }
        
        .chatbot-ctgs-subtitle {
            font-size: 11px;
            color: #666;
        }
        
        .chatbot-ctgs-details-row {
            display: flex;
            gap: 12px;
            margin-top: 4px;
            flex-wrap: wrap;
        }
        
        .chatbot-ctgs-detail-item {
            font-size: 10px;
            color: #555;
            background: #f8f9fa;
            padding: 2px 6px;
            border-radius: 3px;
            border: 1px solid #e9ecef;
        }
        
        .chatbot-ctgs-detail-item strong {
            color: #333;
        }
        
        .chatbot-ctgs-select-btn {
            padding: 6px 12px;
            background: #007bff;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            font-weight: 700;
            transition: all 0.2s ease;
            min-width: 60px;
        }
        
        .chatbot-ctgs-select-btn:hover {
            background: #0056b3;
            transform: scale(1.05);
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }
        
        .chatbot-ctgs-select-btn:disabled {
            background: #ccc;
            cursor: not-allowed;
        }
        
        /* Accounting Table Styles */
        .chatbot-accounting-table {
            margin-top: 15px;
            overflow-x: auto;
            max-width: 100%;
        }
        
        .chatbot-accounting-table table {
            width: 100%;
            border-collapse: collapse;
            font-size: 13px;
            background: white;
            border-radius: 5px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            table-layout: fixed;
        }
        
        .chatbot-accounting-table th {
            background: #28a745;
            color: white;
            padding: 8px 6px;
            text-align: left;
            font-weight: 600;
            font-size: 12px;
            word-wrap: break-word;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        
        .chatbot-accounting-table td {
            padding: 8px 6px;
            border-bottom: 1px solid #eee;
            vertical-align: middle;
            font-size: 12px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        
        .chatbot-accounting-table tr:hover {
            background: #f8f9fa;
        }

        /* Contribution Styles */
        .chatbot-contribution {
            margin: 15px 0;
        }

        .chatbot-contribution-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 12px;
            margin: 4px 0;
            background: #f8f9fa;
            border-radius: 6px;
            border-left: 3px solid #28a745;
        }

        .chatbot-contribution-label {
            font-weight: 500;
            color: #495057;
            font-size: 13px;
        }

        .chatbot-contribution-value {
            font-weight: bold;
            color: #28a745;
            font-size: 13px;
        }

        /* Usage Styles */
        .chatbot-usage {
            margin: 15px 0;
        }

        .chatbot-usage-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 12px;
            margin: 4px 0;
            background: #f8f9fa;
            border-radius: 6px;
            border-left: 3px solid #ffc107;
        }

        .chatbot-usage-label {
            font-weight: 500;
            color: #495057;
            font-size: 13px;
        }

        .chatbot-usage-value {
            font-weight: bold;
            color: #ffc107;
            font-size: 13px;
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .chatbot-overlay {
                bottom: 70px;
                right: 10px;
                width: calc(100vw - 20px);
                height: 70vh;
                max-width: 420px;
            }
            
            .chatbot-message-content {
                max-width: 85%;
            }
            
            .chatbot-popup-icon {
                bottom: 10px;
                right: 10px;
            }
        }

        /* Position variants cho popup icon */
        .chatbot-popup-icon.bottom-left {
            bottom: 20px;
            left: 20px;
            right: auto;
            top: auto;
        }

        .chatbot-popup-icon.top-right {
            top: 20px;
            right: 20px;
            bottom: auto;
            left: auto;
        }

        .chatbot-popup-icon.top-left {
            top: 20px;
            left: 20px;
            bottom: auto;
            right: auto;
        }

        /* Chat overlay luôn ở góc phải, chỉ thay đổi position theo popup icon */
        .chatbot-overlay.top-position {
            top: 90px;
            bottom: auto;
        }

        .chatbot-overlay.left-position {
            left: 20px;
            right: auto;
        }
    `;

    // ChatBot Class
    class ChatBotWidget {
        constructor() {
            this.config = window.CHATBOT_CONFIG;
            this.sessionId = null;
            this.isConnected = false;
            this.isTyping = false;
            this.messageCount = 0;
            this.elements = {};
            this.connectionRetryCount = 0;
            this.maxRetries = 3;
            this.init();
        }

        async init() {
            this.injectStyles();
            this.createElements();
            this.setupEventListeners();
            this.applyConfiguration();
            
            // Lấy thông tin API Mapping (chỉ để log)
            console.log('🗺️ Lấy thông tin API Mapping...');
            await getApiMappingInfo();
            
            // Server ChatBot - động dựa trên domain
            const apiBaseUrl = getApiBaseUrl();
            this.config.apiUrl = apiBaseUrl + SERVER_CONFIG.ENDPOINTS.CHAT;
            console.log('🔗 API URL (động):', this.config.apiUrl);
            console.log('🌐 Current domain:', window.location.hostname);
            
            // Load widget state trước
            const savedState = this.loadWidgetState();
            
            if (savedState && savedState.sessionId) {
                console.log('🔄 Khôi phục widget state với Session ID:', savedState.sessionId);
                this.sessionId = savedState.sessionId;
                
                // Nếu đã kết nối trước đó, thử kết nối lại
                if (savedState.isConnected) {
                    console.log('🔄 Khôi phục trạng thái kết nối');
                    this.isConnected = true;
                    this.elements.messageInput.disabled = false;
                    this.elements.messageInput.placeholder = this.config.placeholder;
                    this.elements.suggestionsBtn.disabled = false;
                    
                    // Ẩn connection status vì đã kết nối
                    this.hideConnectionStatus();
                    
                    // Load lịch sử chat
                    this.loadChatHistory();
                } else {
                    // Test connection với session cũ
                    this.testConnection();
                }
            } else {
                console.log('🆕 Tạo widget state mới...');
                // Tạo session mới
                this.generateSessionId();
                this.testConnection();
            }
            
            // Auto save widget state mỗi 30 giây
            setInterval(() => {
                this.saveWidgetState();
            }, 30000);
            
            // Debug: Kiểm tra localStorage
            console.log('🔍 localStorage hiện tại:', {
                chatbot_session_id: localStorage.getItem('chatbot_session_id'),
                chatbot_widget_state: localStorage.getItem('chatbot_widget_state'),
                allKeys: Object.keys(localStorage)
            });
        }

        injectStyles() {
            if (!document.getElementById('chatbot-styles')) {
                const style = document.createElement('style');
                style.id = 'chatbot-styles';
                style.textContent = CSS_STYLES;
                document.head.appendChild(style);
            }
        }

        createElements() {
            // Create popup icon
            this.elements.popupIcon = document.createElement('div');
            this.elements.popupIcon.className = 'chatbot-popup-icon';
            this.elements.popupIcon.innerHTML = `
                <i class="fas fa-comments"></i>
                <div class="chatbot-notification-badge" style="display: none;">0</div>
            `;

            // Create chat overlay
            this.elements.overlay = document.createElement('div');
            this.elements.overlay.className = 'chatbot-overlay';
            this.elements.overlay.innerHTML = `
                <div class="chatbot-container">
                    <div class="chatbot-header">
                        <div class="chatbot-logo">
                            <i class="fas fa-robot"></i>
                            <h1 class="chatbot-title">${this.config.title}</h1>
                        </div>
                        <div class="chatbot-session-info">
                            <button class="chatbot-btn chatbot-new-session-btn">
                                <i class="fas fa-plus"></i> Phiên mới
                            </button>
                            <button class="chatbot-btn chatbot-reset-btn" title="Reset widget">
                                <i class="fas fa-redo"></i>
                            </button>
                            <button class="chatbot-btn-close chatbot-close-btn">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>

                    <div class="chatbot-body">
                        <div class="chatbot-connection-status">
                            <div class="chatbot-connection-status-icon">
                                <i class="fas fa-satellite-dish"></i>
                            </div>
                            <div class="chatbot-connection-status-text">Đang kết nối...</div>
                            <div class="chatbot-connection-status-subtitle">Vui lòng đợi trong giây lát</div>
                        </div>

                        <div class="chatbot-messages">
                            <!-- Welcome message sẽ được thêm sau khi kết nối thành công -->
                        </div>

                        <div class="chatbot-typing-indicator">
                            <div class="chatbot-typing-avatar">
                                <i class="fas fa-robot"></i>
                            </div>
                            <span>Đang nhập...</span>
                            <div class="chatbot-typing-dots">
                                <div class="chatbot-typing-dot"></div>
                                <div class="chatbot-typing-dot"></div>
                                <div class="chatbot-typing-dot"></div>
                            </div>
                        </div>

                        <div class="chatbot-input-container">
                            <div class="chatbot-input-wrapper">
                                <textarea 
                                    class="chatbot-message-input" 
                                    placeholder="Đang kết nối..."
                                    rows="1"
                                    autocomplete="off"
                                    autocorrect="off"
                                    autocapitalize="sentences"
                                    disabled
                                ></textarea>
                                <button class="chatbot-suggestions-btn" title="Gợi ý chức năng" disabled>
                                    <i class="fas fa-plus"></i>
                                </button>
                                <button class="chatbot-send-btn" disabled>
                                    <i class="fas fa-paper-plane"></i>
                                </button>
                            </div>
                            
                            <!-- Suggestions Menu -->
                            <div class="chatbot-suggestions-menu">
                                <div class="chatbot-suggestions-header">
                                    <i class="fas fa-lightbulb"></i>
                                    <span>Gợi ý chức năng</span>
                                </div>
                                
                                <!-- Tab Navigation -->
                                <div class="chatbot-suggestions-tabs">
                                    <button class="chatbot-suggestion-tab active" data-tab="tao-chung-tu">
                                        <i class="fas fa-plus-circle"></i>
                                        Tạo chứng từ
                                    </button>
                                    <button class="chatbot-suggestion-tab" data-tab="tra-cuu">
                                        <i class="fas fa-search"></i>
                                        Tra cứu
                                    </button>
                                </div>
                                
                                <!-- Tab Content -->
                                <div class="chatbot-suggestions-content">
                                    <!-- Tab: Tạo chứng từ -->
                                    <div class="chatbot-suggestion-pane active" id="tao-chung-tu">
                                        <div class="chatbot-suggestions-list">
                                            <button class="chatbot-suggestion-item" data-suggestion="Tạo lương tháng">
                                                <i class="fas fa-money-bill-wave"></i>
                                                <span>Tạo lương tháng</span>
                                            </button>
                                            <button class="chatbot-suggestion-item" data-suggestion="Tạo bảo hiểm tháng">
                                                <i class="fas fa-shield-alt"></i>
                                                <span>Tạo bảo hiểm tháng</span>
                                            </button>
                                            <button class="chatbot-suggestion-item" data-suggestion="Tạo kinh phí công đoàn tháng">
                                                <i class="fas fa-handshake"></i>
                                                <span>Tạo kinh phí công đoàn tháng</span>
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <!-- Tab: Tra cứu -->
                                    <div class="chatbot-suggestion-pane" id="tra-cuu">
                                        <div class="chatbot-suggestions-list">
                                            <button class="chatbot-suggestion-item" data-suggestion="Tra cứu lương">
                                                <i class="fas fa-search-dollar"></i>
                                                <span>Tra cứu lương</span>
                                            </button>
                                            <button class="chatbot-suggestion-item" data-suggestion="Tra cứu bảo hiểm">
                                                <i class="fas fa-search-shield"></i>
                                                <span>Tra cứu bảo hiểm</span>
                                            </button>
                                            <button class="chatbot-suggestion-item" data-suggestion="Tra cứu CTGS">
                                                <i class="fas fa-file-invoice"></i>
                                                <span>Tra cứu CTGS</span>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            // Add elements to DOM
            document.body.appendChild(this.elements.popupIcon);
            document.body.appendChild(this.elements.overlay);

            // Cache elements
            this.elements.messages = this.elements.overlay.querySelector('.chatbot-messages');
            this.elements.messageInput = this.elements.overlay.querySelector('.chatbot-message-input');
            this.elements.sendBtn = this.elements.overlay.querySelector('.chatbot-send-btn');
            this.elements.suggestionsBtn = this.elements.overlay.querySelector('.chatbot-suggestions-btn');
            this.elements.suggestionsMenu = this.elements.overlay.querySelector('.chatbot-suggestions-menu');
            this.elements.inputContainer = this.elements.overlay.querySelector('.chatbot-input-container');
            this.elements.typingIndicator = this.elements.overlay.querySelector('.chatbot-typing-indicator');
            this.elements.connectionStatus = this.elements.overlay.querySelector('.chatbot-connection-status');
        }

        setupEventListeners() {
            // Popup icon click
            this.elements.popupIcon.addEventListener('click', () => this.open());

            // Close button click
            this.elements.overlay.querySelector('.chatbot-close-btn').addEventListener('click', () => this.close());

            // New session button click
            this.elements.overlay.querySelector('.chatbot-new-session-btn').addEventListener('click', () => this.startNewSession());

            // Reset button click
            this.elements.overlay.querySelector('.chatbot-reset-btn').addEventListener('click', () => this.resetWidget());

            // Message input events
            this.elements.messageInput.addEventListener('input', () => this.handleInputChange());
            this.elements.messageInput.addEventListener('keydown', (e) => this.handleKeyDown(e));

            // Send button click
            this.elements.sendBtn.addEventListener('click', () => this.sendMessage());

            // Suggestions button click
            this.elements.suggestionsBtn.addEventListener('click', () => this.toggleSuggestions());

            // Suggestions items click
            this.elements.suggestionsMenu.addEventListener('click', (e) => {
                if (e.target.closest('.chatbot-suggestion-item')) {
                    const item = e.target.closest('.chatbot-suggestion-item');
                    
                    // Kiểm tra nếu button đang disabled thì không cho click
                    if (item.disabled) {
                        return;
                    }
                    
                    const suggestion = item.dataset.suggestion;
                    
                    // Disable tất cả suggestion buttons khi bắt đầu xử lý
                    const allSuggestions = this.elements.suggestionsMenu.querySelectorAll('.chatbot-suggestion-item');
                    allSuggestions.forEach(btn => {
                        btn.disabled = true;
                        btn.style.opacity = '0.6';
                    });
                    
                    // Thay đổi text của button được click
                    const originalText = item.innerHTML;
                    item.innerHTML = '<i class="fas fa-spinner fa-spin"></i> <span>Đang xử lý...</span>';
                    
                    this.useSuggestion(suggestion).finally(() => {
                        // Re-enable tất cả buttons sau khi hoàn thành
                        allSuggestions.forEach(btn => {
                            btn.disabled = false;
                            btn.style.opacity = '1';
                        });
                        
                        // Khôi phục text gốc
                        item.innerHTML = originalText;
                    });
                }
            });

            // Close suggestions when clicking outside
            document.addEventListener('click', (e) => {
                if (!this.elements.inputContainer.contains(e.target)) {
                    this.hideSuggestions();
                }
            });

            // Không đóng chat khi click outside (để có thể thao tác với website)
            // this.elements.overlay.addEventListener('click', (e) => {
            //     if (e.target === this.elements.overlay) {
            //         this.close();
            //     }
            // });
        }

        applyConfiguration() {
            // Apply theme
            const root = document.documentElement;
            root.style.setProperty('--primary-color', this.config.theme.primaryColor);
            root.style.setProperty('--secondary-color', this.config.theme.secondaryColor);
            root.style.setProperty('--background-color', this.config.theme.backgroundColor);
            root.style.setProperty('--text-color', this.config.theme.textColor);

            // Apply position
            this.elements.popupIcon.className = `chatbot-popup-icon ${this.config.position}`;

            // Show/hide popup icon
            this.elements.popupIcon.style.display = this.config.showPopupIcon ? 'flex' : 'none';

            // Auto open if configured
            if (this.config.autoOpen) {
                setTimeout(() => this.open(), 1000);
            }
        }

        generateSessionId() {
            const timestamp = Date.now();
            const random = Math.random().toString(36).substring(2, 15);
            this.sessionId = `session_${timestamp}_${random}`;
            this.saveSessionId();
            this.saveWidgetState();
            console.log('ChatBot Session ID:', this.sessionId);
        }

        saveSessionId() {
            if (this.sessionId) {
                try {
                    localStorage.setItem('chatbot_session_id', this.sessionId);
                    console.log('💾 Đã lưu Session ID vào localStorage:', this.sessionId);
                    
                    // Verify save
                    const saved = localStorage.getItem('chatbot_session_id');
                    if (saved === this.sessionId) {
                        console.log(' Xác nhận lưu thành công');
                    } else {
                        console.error('❌ Lỗi khi lưu Session ID');
                    }
                } catch (error) {
                    console.error('❌ Lỗi khi lưu Session ID vào localStorage:', error);
                }
            } else {
                console.warn('⚠️ Không có Session ID để lưu');
            }
        }

        saveWidgetState() {
            try {
                const widgetState = {
                    sessionId: this.sessionId,
                    isConnected: this.isConnected,
                    lastActivity: Date.now(),
                    config: this.config
                };
                
                localStorage.setItem('chatbot_widget_state', JSON.stringify(widgetState));
                console.log('💾 Đã lưu widget state:', widgetState);
            } catch (error) {
                console.error('❌ Lỗi khi lưu widget state:', error);
            }
        }

        loadWidgetState() {
            try {
                const savedState = localStorage.getItem('chatbot_widget_state');
                if (savedState) {
                    const state = JSON.parse(savedState);
                    console.log('📂 Đã load widget state:', state);
                    
                    // Kiểm tra xem state có hợp lệ không
                    const now = Date.now();
                    const lastActivity = state.lastActivity || 0;
                    const maxAge = 24 * 60 * 60 * 1000; // 24 giờ
                    
                    // Nếu state quá cũ, xóa nó
                    if (now - lastActivity > maxAge) {
                        console.log('🗑️ Widget state quá cũ, xóa bỏ');
                        localStorage.removeItem('chatbot_widget_state');
                        return null;
                    }
                    
                    // Khôi phục session ID nếu có
                    if (state.sessionId) {
                        this.sessionId = state.sessionId;
                    }
                    
                    // Khôi phục trạng thái kết nối
                    if (state.isConnected) {
                        this.isConnected = state.isConnected;
                    }
                    
                    return state;
                }
                return null;
            } catch (error) {
                console.error('❌ Lỗi khi load widget state:', error);
                // Xóa widget state bị lỗi
                localStorage.removeItem('chatbot_widget_state');
                return null;
            }
        }

        loadSessionId() {
            try {
                const savedSessionId = localStorage.getItem('chatbot_session_id');
                if (savedSessionId && savedSessionId.trim() !== '') {
                    this.sessionId = savedSessionId;
                    console.log('📂 Đã khôi phục Session ID từ localStorage:', this.sessionId);
                    return true;
                }
                console.log('❌ Không tìm thấy Session ID trong localStorage');
                return false;
            } catch (error) {
                console.error('❌ Lỗi khi load Session ID từ localStorage:', error);
                return false;
            }
        }

        clearSessionId() {
            localStorage.removeItem('chatbot_session_id');
            console.log('🗑️ Đã xóa Session ID khỏi localStorage');
        }

        clearWidgetState() {
            localStorage.removeItem('chatbot_widget_state');
            console.log('🗑️ Đã xóa Widget State khỏi localStorage');
        }

        resetWidget() {
            console.log('🔄 Reset widget hoàn toàn...');
            this.clearSessionId();
            this.clearWidgetState();
            this.isConnected = false;
            this.connectionRetryCount = 0;
            this.sessionId = null;
            
            // Reset UI
            this.elements.messageInput.disabled = true;
            this.elements.messageInput.placeholder = 'Đang kết nối...';
            this.elements.suggestionsBtn.disabled = true;
            this.elements.messages.innerHTML = '';
            
            // Ẩn connection status
            this.hideConnectionStatus();
            
            // Tạo session mới
            this.generateSessionId();
            this.testConnection();
        }

        // Debug function để xóa localStorage
        debugClearAll() {
            console.log('🧹 Debug: Xóa tất cả localStorage liên quan đến ChatBot');
            localStorage.removeItem('chatbot_session_id');
            localStorage.removeItem('chatbot_widget_state');
            console.log(' Đã xóa tất cả localStorage');
            console.log('🔍 localStorage hiện tại:', Object.keys(localStorage));
        }

        async testConnection() {
            // Đảm bảo có sessionId
            if (!this.sessionId) {
                this.generateSessionId();
            }
            
            try {
                const controller = new AbortController();
                const timeoutId = setTimeout(() => controller.abort(), 2000); // Giảm timeout xuống 2 giây
                
                const response = await fetch(this.config.apiUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        message: 'test',
                        sessionId: this.sessionId
                    }),
                    signal: controller.signal
                });

                clearTimeout(timeoutId);

                if (response.ok) {
                    this.isConnected = true;
                    this.connectionRetryCount = 0; // Reset retry count khi thành công
                    // Ẩn connection status và enable input ngay lập tức
                    this.hideConnectionStatus();
                    this.elements.messageInput.disabled = false;
                    this.elements.messageInput.placeholder = this.config.placeholder;
                    this.elements.suggestionsBtn.disabled = false;
                    
                    // Thử load lịch sử chat trước khi hiển thị welcome message
                    await this.loadChatHistory();
                    
                    console.log(' ChatBot kết nối thành công với Session ID:', this.sessionId);
                } else {
                    throw new Error(`HTTP ${response.status}`);
                }
            } catch (error) {
                this.connectionRetryCount++;
                
                if (error.name === 'AbortError') {
                    console.error(`❌ ChatBot timeout kết nối (2s) - Lần thử ${this.connectionRetryCount}/${this.maxRetries}`);
                } else {
                    console.error(`❌ ChatBot lỗi kết nối: ${error} - Lần thử ${this.connectionRetryCount}/${this.maxRetries}`);
                }
                
                // Chỉ hiển thị connection status nếu chat đang mở và chưa kết nối
                if (this.elements.overlay.style.display === 'block' && !this.isConnected) {
                    this.showConnectionStatus();
                }
                
                // Chỉ retry nếu chưa vượt quá số lần thử tối đa
                if (this.elements.overlay.style.display === 'block' && !this.isConnected && this.connectionRetryCount < this.maxRetries) {
                    // Giảm thời gian retry xuống 1.5 giây
                    setTimeout(() => this.testConnection(), 1500);
                } else if (this.connectionRetryCount >= this.maxRetries) {
                    // Dừng retry và hiển thị thông báo lỗi
                    console.error('❌ Đã thử kết nối tối đa số lần, dừng retry');
                    this.hideConnectionStatus();
                    this.elements.messageInput.placeholder = 'Nhấn Enter để thử lại...';
                }
            }
        }

        open() {
            // Ẩn popup icon trước
            this.elements.popupIcon.style.display = 'none';
            this.elements.popupIcon.style.visibility = 'hidden';
            
            // Hiển thị chat overlay
            this.elements.overlay.style.display = 'block';

            if (this.config.autoFocus && this.isConnected) {
                setTimeout(() => {
                    this.elements.messageInput.focus();
                }, 100);
            }

            if (this.config.onOpen && typeof this.config.onOpen === 'function') {
                this.config.onOpen();
            }
        }

        close() {
            // Ẩn chat overlay
            this.elements.overlay.style.display = 'none';
            
            // Hiển thị lại popup icon
            if (this.config.showPopupIcon) {
                this.elements.popupIcon.style.display = 'flex';
                this.elements.popupIcon.style.visibility = 'visible';
            }

            if (this.config.onClose && typeof this.config.onClose === 'function') {
                this.config.onClose();
            }
        }

        startNewSession() {
            // Xóa session cũ khỏi localStorage
            this.clearSessionId();
            
            // Tạo session mới
            this.generateSessionId();
            this.isConnected = false;
            this.connectionRetryCount = 0; // Reset retry count
            
            // Xóa tất cả messages, chỉ hiển thị connection status
            this.elements.messages.innerHTML = '';
            
            // Disable input và hiển thị connection status
            this.elements.messageInput.disabled = true;
            this.elements.messageInput.placeholder = 'Đang kết nối...';
            this.elements.suggestionsBtn.disabled = true;
            this.showConnectionStatus();
            
            // Test connection ngay lập tức
            this.testConnection();
            
            // Tự động ẩn connection status sau 3 giây nếu vẫn chưa kết nối được
            setTimeout(() => {
                if (!this.isConnected) {
                    this.hideConnectionStatus();
                    this.elements.messageInput.placeholder = 'Nhấn Enter để thử lại...';
                }
            }, 3000);
        }

        handleInputChange() {
            const message = this.elements.messageInput.value.trim();
            this.elements.sendBtn.disabled = !message;
            
            // Auto-resize textarea
            this.elements.messageInput.style.height = 'auto';
            this.elements.messageInput.style.height = Math.min(this.elements.messageInput.scrollHeight, 100) + 'px';
        }

        handleKeyDown(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                
                // Nếu chưa kết nối và placeholder là "Nhấn Enter để thử lại..."
                if (!this.isConnected && this.elements.messageInput.placeholder === 'Nhấn Enter để thử lại...') {
                    this.startNewSession();
                    return;
                }
                
                if (!this.elements.sendBtn.disabled) {
                    this.sendMessage();
                }
            }
        }

        async sendMessage() {
            const message = this.elements.messageInput.value.trim();
            if (!message || !this.isConnected) return;

            // Sử dụng sendMessageWithFlag với isApiCall = false
                            this.guiTinNhanCoCo(message, false);
        }

        addMessage(text, sender, isApiCall = false) {
            const messageDiv = document.createElement('div');
            messageDiv.className = `chatbot-message ${sender}`;
            if (isApiCall) {
                messageDiv.classList.add('api-call');
            }
            
            const avatar = document.createElement('div');
            avatar.className = 'chatbot-message-avatar';
            avatar.innerHTML = sender === 'user' ? '<i class="fas fa-user"></i>' : '<i class="fas fa-robot"></i>';
            
            const content = document.createElement('div');
            content.className = 'chatbot-message-content';
            
            // Kiểm tra nếu text chứa HTML
            const safeText = String(text || '');
            if (safeText.includes('<div') || safeText.includes('<table') || safeText.includes('<button')) {
                content.innerHTML = safeText;
            } else {
                content.textContent = safeText;
            }
            
            // Thêm badge API call nếu cần
            if (isApiCall && sender === 'user') {
                const apiBadge = document.createElement('div');
                apiBadge.className = 'chatbot-api-badge';
                apiBadge.innerHTML = '<i class="fas fa-code"></i> API';
                content.appendChild(apiBadge);
            }
            
            messageDiv.appendChild(avatar);
            messageDiv.appendChild(content);
            
            this.elements.messages.appendChild(messageDiv);
            this.elements.messages.scrollTop = this.elements.messages.scrollHeight;
        }

        showTypingIndicator() {
            this.elements.typingIndicator.classList.add('show');
            this.elements.messages.scrollTop = this.elements.messages.scrollHeight;
        }

        hideTypingIndicator() {
            this.elements.typingIndicator.classList.remove('show');
        }

        showConnectionStatus() {
            this.elements.connectionStatus.style.display = 'flex';
            // Giữ messages hiển thị phía sau connection status
        }

        hideConnectionStatus() {
            this.elements.connectionStatus.style.display = 'none';
            // Messages luôn hiển thị
        }

        showWelcomeMessage() {
            this.elements.messages.innerHTML = `
                <div class="chatbot-welcome-message">
                    <div class="chatbot-welcome-icon">
                        <i class="fas fa-comments"></i>
                    </div>
                    <h2>Chào mừng bạn!</h2>
                    <p class="chatbot-welcome-text">${this.config.welcomeMessage}</p>
                    
                    <!-- Tab Navigation -->
                </div>
            `;
            
            // Add tab functionality
            this.initTabs();
        }

        updateSessionDisplay() {
            // Cập nhật hiển thị session ID trong console
            if (this.sessionId) {
                console.log('📋 Session hiện tại:', this.sessionId);
            }
        }

        initTabs() {
            const tabButtons = document.querySelectorAll('.chatbot-tab-btn');
            const tabPanes = document.querySelectorAll('.chatbot-tab-pane');
            
            tabButtons.forEach(button => {
                button.addEventListener('click', () => {
                    const targetTab = button.getAttribute('data-tab');
                    
                    // Remove active class from all buttons and panes
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    tabPanes.forEach(pane => pane.classList.remove('active'));
                    
                    // Add active class to clicked button and corresponding pane
                    button.classList.add('active');
                    document.getElementById(targetTab).classList.add('active');
                });
            });
        }

        initSuggestionsTabs() {
            const tabButtons = this.elements.suggestionsMenu.querySelectorAll('.chatbot-suggestion-tab');
            const tabPanes = this.elements.suggestionsMenu.querySelectorAll('.chatbot-suggestion-pane');
            
            tabButtons.forEach(button => {
                button.addEventListener('click', (e) => {
                    e.stopPropagation(); // Prevent triggering suggestion item click
                    
                    const targetTab = button.getAttribute('data-tab');
                    
                    // Remove active class from all buttons and panes
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    tabPanes.forEach(pane => pane.classList.remove('active'));
                    
                    // Add active class to clicked button and corresponding pane
                    button.classList.add('active');
                    this.elements.suggestionsMenu.querySelector(`#${targetTab}`).classList.add('active');
                });
            });
        }

        async loadChatHistory() {
            try {
                console.log('📚 Đang tải lịch sử chat...');
                
                const response = await fetch(getApiUrl(`/api/session/${this.sessionId}/messages`));
                
                if (response.ok) {
                    const data = await response.json();
                    const messages = data.messages || [];
                    
                    if (messages.length > 0) {
                        console.log(`📚 Đã tải ${messages.length} tin nhắn từ lịch sử`);
                        
                        // Xóa welcome message nếu có
                        this.elements.messages.innerHTML = '';
                        
                        // Hiển thị tất cả tin nhắn từ lịch sử
                        messages.forEach(msg => {
                            if (msg.role === 'user' || msg.role === 'assistant') {
                                this.addMessage(msg.content, msg.role);
                            }
                        });
                        
                        // Thêm thông báo "Đã khôi phục phiên chat"
                        this.addSystemMessage('🔄 Đã khôi phục phiên chat trước đó');
                        
                        // Scroll xuống tin nhắn cuối cùng sau khi tất cả tin nhắn đã được thêm
                        setTimeout(() => {
                            this.elements.messages.scrollTop = this.elements.messages.scrollHeight;
                        }, 100);
                        
                    } else {
                        console.log('📚 Không có lịch sử chat, hiển thị welcome message');
                        this.showWelcomeMessage();
                    }
                } else {
                    console.log('📚 Không thể tải lịch sử chat, hiển thị welcome message');
                    this.showWelcomeMessage();
                }
            } catch (error) {
                console.error('❌ Lỗi khi tải lịch sử chat:', error);
                this.showWelcomeMessage();
            }
        }

        addSystemMessage(text) {
            const messageDiv = document.createElement('div');
            messageDiv.className = 'chatbot-system-message';
            messageDiv.innerHTML = `
                <div class="chatbot-system-content">
                    <i class="fas fa-info-circle"></i>
                    <span>${text}</span>
                </div>
            `;
            this.elements.messages.appendChild(messageDiv);
            this.elements.messages.scrollTop = this.elements.messages.scrollHeight;
        }

        toggleSuggestions() {
            console.log('🔍 Toggle suggestions menu');
            console.log('Menu element:', this.elements.suggestionsMenu);
            console.log('Current classes:', this.elements.suggestionsMenu.className);
            
            if (this.elements.suggestionsMenu.classList.contains('show')) {
                this.hideSuggestions();
            } else {
                this.showSuggestions();
            }
        }

        showSuggestions() {
            console.log('📋 Hiển thị suggestions menu');
            this.elements.suggestionsMenu.classList.add('show');
            console.log('Classes sau khi thêm show:', this.elements.suggestionsMenu.className);
            
            // Initialize suggestions tabs
            this.initSuggestionsTabs();
        }

        hideSuggestions() {
            this.elements.suggestionsMenu.classList.remove('show');
        }

        async useSuggestion(suggestion) {
            console.log('💡 Sử dụng gợi ý:', suggestion);
            
            // Ẩn suggestions menu
            this.hideSuggestions();
            
            // Add user message first
            this.addMessage(suggestion, 'user', true);
            
            // Show typing indicator
            this.showTypingIndicator();
            
            // Add a small delay to ensure typing indicator is visible
            await new Promise(resolve => setTimeout(resolve, 300));
            
            try {
                let result;
                
                switch (suggestion) {
                    case 'Tạo lương tháng':
                        // Lưu context cho lương
                        localStorage.setItem('currentContext', 'Tạo lương tháng');
                        result = await window.ChatBotAPI.createSalaryWithCTGS();
                        break;
                    case 'Tạo bảo hiểm tháng':
                        // Lưu context cho bảo hiểm
                        localStorage.setItem('currentContext', 'Tạo bảo hiểm tháng');
                        result = await window.ChatBotAPI.createInsuranceWithCTGS();
                        break;
                    case 'Tạo kinh phí công đoàn tháng':
                        // Lưu context cho công đoàn
                        localStorage.setItem('currentContext', 'Tạo kinh phí công đoàn tháng');
                        result = await window.ChatBotAPI.createUnionWithCTGS();
                        break;
                    default:
                        // Fallback to old method for other suggestions
                        this.hideTypingIndicator();
                        // Lưu context cho các lệnh khác
                        localStorage.setItem('currentContext', suggestion);
                        this.guiTinNhanCoCo(suggestion, true);
                        return Promise.resolve();
                }
                
                console.log('📥 Frontend nhận được result:', result);
                if (result && result.success) {
                    const data = result.data;
                    console.log('📋 Data từ server:', data);
                    console.log('🔍 Data type:', data.type);
                    let responseMessage = '';
                    
                    switch (data.type) {
                        case 'ctgs_create_form':
                            // Hiển thị form tạo CTGS
                            responseMessage = data.html || 'Không thể hiển thị form tạo CTGS';
                            break;
                            
                        case 'ctgs_selection':
                            // Hiển thị thông tin chi tiết hạch toán trước
                            responseMessage = '';
                            
                            if (data.accountingDetails && data.accountingDetails.data) {
                                responseMessage += `<div style="font-size: 16px; font-weight: bold; margin-bottom: 10px;">Chi tiết hạch toán</div>
<div class="chatbot-accounting-table">
    <table>
        <thead>
            <tr>
                <th>T.Mục</th>
                <th>Nội dung</th>
                <th>Số tiền (VNĐ)</th>
                <th>Nguồn</th>
            </tr>
        </thead>
        <tbody>
            ${data.accountingDetails.data.map((item, index) => `
                <tr>
                    <td>${item.Smtm ? item.Smtm.slice(-4) : 'N/A'}</td>
                    <td>${item.Ndctcts || 'N/A'}</td>
                    <td>${item.Dst ? item.Dst.toLocaleString('vi-VN') : 'N/A'}</td>
                    <td>${item.Smn || 'N/A'}</td>
                </tr>
            `).join('')}
            <tr style="background: #e8f5e8; font-weight: bold;">
                <td colspan="2" style="text-align: right;"><strong>Tổng cộng:</strong></td>
                <td style="color: #28a745; white-space: nowrap;"><strong>${data.accountingDetails.summary && data.accountingDetails.summary[0] ? data.accountingDetails.summary[0].toLocaleString('vi-VN') : 'N/A'}</strong></td>
                <td></td>
            </tr>
        </tbody>
    </table>
</div>

`;
                            }
                            
                            // Sau đó hiển thị bảng CTGS để chọn
                            const currentContext = localStorage.getItem('currentContext') || 'Tạo lương tháng';
                            const actionText = currentContext.includes('bảo hiểm') ? 'tạo bảo hiểm' : 
                                             currentContext.includes('công đoàn') ? 'tạo công đoàn' : 'tạo lương';
                            
                            responseMessage += `<div style="font-size: 16px; font-weight: bold; margin: 20px 0 10px 0;">Chọn một CTGS để ${actionText}</div>

<div class="chatbot-accounting-table">
    <table>
        <thead>
            <tr>
                <th>Số hiệu</th>
                <th>Kế toán</th>
                <th>Thao tác</th>
            </tr>
        </thead>
        <tbody>
            ${data.ctgsList.map(ctgs => `
                <tr>
                    <td>${ctgs.soHieu}</td>
                    <td>${ctgs.keToan}</td>
                    <td>
                        <button class="chatbot-ctgs-select-btn" onclick="window.ChatBotAPI.selectCTGS(${ctgs.index})">
                            Chọn
                        </button>
                    </td>
                </tr>
            `).join('')}
        </tbody>
    </table>
</div>`;
                            break;
                            
                        case 'salary':
                        default:
                            // Xử lý response tạo lương thành công (không có type hoặc type = 'salary')
                            if (data.ctgsId && data.apiResult) {
                                // Hiển thị thông tin đơn giản khi thành công
                                responseMessage = `<div style="font-size: 16px; font-weight: bold; margin-bottom: 10px;">Tạo lương tháng ${data.month}/${data.year} thành công!</div>

<div class="chatbot-action-buttons">
    <button class="chatbot-action-btn primary" onclick="window.ChatBotAPI.viewSalaryDetails('${data.month}', '${data.year}')">
        Xem chi tiết
    </button>
</div>`;
                            } else {
                                // Fallback cho response không có cấu trúc đúng
                                responseMessage = `<div style="font-size: 16px; font-weight: bold; margin-bottom: 10px;">Tạo lương thành công!</div>

<div style="font-size: 14px; font-weight: bold; margin-bottom: 5px;">Thông tin:</div>
- Dữ liệu: ${JSON.stringify(data, null, 2)}
- Trạng thái: Thành công`;
                            }
                            break;
                            

                            
                        case 'insurance':
                            responseMessage = `🛡️ **Bảng bảo hiểm tháng ${data.month}/${data.year}** (CTGS ID: ${data.ctgsId})

<div class="chatbot-table-container">
  <table class="chatbot-table">
    <thead>
      <tr>
        <th>Loại bảo hiểm</th>
        <th>Số tiền (VNĐ)</th>
      </tr>
    </thead>
    <tbody>
      <tr>
        <td>📊 Tổng nhân viên</td>
        <td><strong>${data.totalEmployees || 'Đang cập nhật'} người</strong></td>
      </tr>
      <tr>
        <td>🏥 Bảo hiểm xã hội</td>
        <td><strong>${data.socialInsurance ? data.socialInsurance.toLocaleString('vi-VN') : 'Đang tính toán'}</strong></td>
      </tr>
      <tr>
        <td>💊 Bảo hiểm y tế</td>
        <td><strong>${data.healthInsurance ? data.healthInsurance.toLocaleString('vi-VN') : 'Đang tính toán'}</strong></td>
      </tr>
      <tr>
        <td>🛡️ Bảo hiểm thất nghiệp</td>
        <td><strong>${data.unemploymentInsurance ? data.unemploymentInsurance.toLocaleString('vi-VN') : 'Đang tính toán'}</strong></td>
      </tr>
      <tr style="background: #e8f5e8; font-weight: bold;">
        <td>💰 Tổng cộng</td>
        <td><strong style="color: #28a745;">${(data.socialInsurance && data.healthInsurance && data.unemploymentInsurance) ? (data.socialInsurance + data.healthInsurance + data.unemploymentInsurance).toLocaleString('vi-VN') : 'Đang tính toán'}</strong></td>
      </tr>
      <tr>
        <td>📋 CTGS ID</td>
        <td><strong>${data.ctgsId}</strong></td>
      </tr>
    </tbody>
  </table>
</div>

<div class="chatbot-action-buttons">
  <button class="chatbot-action-btn primary" onclick="ChatBotAPI.viewFile('insurance', '${data.month}', '${data.year}')">
    📁 Xem file Excel
  </button>
  <button class="chatbot-action-btn secondary" onclick="ChatBotAPI.viewDetails('insurance')">
    📊 Xem chi tiết
  </button>
  <button class="chatbot-action-btn success" onclick="ChatBotAPI.downloadFile('insurance', '${data.month}', '${data.year}')">
    ⬇️ Tải xuống
  </button>
</div>

 **Trạng thái:** Đã tạo thành công với CTGS
📁 **File:** ${data.fileName}
⏱️ **Thời gian xử lý:** ${result.processingTime}ms`;
                            break;
                            
                        case 'union':
                            responseMessage = `👥 **Bảng kinh phí công đoàn tháng ${data.month}/${data.year}** (CTGS ID: ${data.ctgsId})

<div class="chatbot-table-container">
  <table class="chatbot-table">
    <thead>
      <tr>
        <th>Thông tin</th>
        <th>Giá trị</th>
      </tr>
    </thead>
    <tbody>
      <tr>
        <td>📊 Tổng nhân viên</td>
        <td><strong>${data.totalEmployees || 'Đang cập nhật'} người</strong></td>
      </tr>
      <tr>
        <td>💰 Kinh phí công đoàn</td>
        <td><strong>${data.unionFund ? data.unionFund.toLocaleString('vi-VN') : 'Đang tính toán'} VNĐ</strong></td>
      </tr>
      <tr>
        <td>📈 Tỷ lệ đóng góp</td>
        <td><strong>${data.contributionRate || 'Đang cập nhật'}</strong></td>
      </tr>
      <tr>
        <td>📋 CTGS ID</td>
        <td><strong>${data.ctgsId}</strong></td>
      </tr>
    </tbody>
  </table>
</div>

<div class="chatbot-action-buttons">
  <button class="chatbot-action-btn primary" onclick="ChatBotAPI.viewFile('union', '${data.month}', '${data.year}')">
    📁 Xem file Excel
  </button>
  <button class="chatbot-action-btn secondary" onclick="ChatBotAPI.viewDetails('union')">
    📊 Xem chi tiết
  </button>
  <button class="chatbot-action-btn success" onclick="ChatBotAPI.downloadFile('union', '${data.month}', '${data.year}')">
    ⬇️ Tải xuống
  </button>
</div>

 **Trạng thái:** Đã tạo thành công với CTGS
📁 **File:** ${data.fileName}
⏱️ **Thời gian xử lý:** ${result.processingTime}ms`;
                            break;
                    }
                    
                    console.log('📝 Response message sẽ hiển thị:', responseMessage);
                    
                    // Add bot response
                    this.addMessage(responseMessage, 'bot');
                    
                    // Hide typing indicator after a short delay
                    setTimeout(() => {
                        this.hideTypingIndicator();
                    }, 500);
                    
                } else {
                    throw new Error(result?.error || 'Lỗi không xác định');
                }
                
            } catch (error) {
                console.error('❌ Error in useSuggestion:', error);
                
                // Add error message
                this.addMessage(`<div style="color: #dc3545; font-weight: bold;">Lỗi: ${error.message}</div>`, 'bot');
                
                // Hide typing indicator after a short delay
                setTimeout(() => {
                    this.hideTypingIndicator();
                }, 500);
            } finally {
                // Đảm bảo luôn return Promise
                return Promise.resolve();
            }
        }

        guiTinNhanCoCo(message, isApiCall = false) {
            if (!message || !this.isConnected) return;

            // Tạo sessionId nếu chưa có
            if (!this.sessionId) {
                this.generateSessionId();
            }

            // Ẩn welcome message khi bắt đầu chat
            const welcomeMessage = this.elements.messages.querySelector('.chatbot-welcome-message');
            if (welcomeMessage) {
                welcomeMessage.style.display = 'none';
            }

            // Add user message với flag API call
            this.addMessage(message, 'user', isApiCall);
            this.elements.messageInput.value = '';
            this.handleInputChange();

            // Show typing indicator
            this.showTypingIndicator();

            // Gửi tin nhắn với flag API call
            this.sendMessageToServer(message, isApiCall);
        }

        async sendMessageToServer(message, isApiCall = false) {
            try {
                // Lấy API data từ localStorage
                const apiData = window.ChatBotAPI.layDuLieuApi();
                
                // Kiểm tra token_httc bắt buộc
                if (!apiData || !apiData.token_httc) {
                    console.error('❌ Thiếu token_httc trong API data');
                    this.hideTypingIndicator();
                    this.addMessage('<div style="color: #dc3545; font-weight: bold;">Lỗi: Thiếu token_httc. Vui lòng cấu hình API data trước.</div>', 'bot');
                    return;
                }
                
                // Tự động thêm "Bearer" vào token nếu chưa có
                const tokenWithBearer = apiData.token_httc.startsWith('Bearer ') 
                    ? apiData.token_httc 
                    : `Bearer ${apiData.token_httc}`;
                
                // Lấy thông tin domain và user từ localStorage
                const userFromStorage = localStorage.getItem('user');
                const userFromApiData = apiData.user;
                const finalUser = userFromStorage || userFromApiData || 'anonymous';
                
                console.log('🔍 Debug User Info:', {
                    userFromStorage: userFromStorage,
                    userFromApiData: userFromApiData,
                    finalUser: finalUser,
                    apiData: apiData
                });
                
                const userInfo = {
                    user_id: finalUser,
                    domain: window.location.hostname,
                    user: finalUser,
                    token_httc: apiData.token_httc || '',
                    kthcMaCoQuan: apiData.kthcMaCoQuan || null,
                    kthcMaCanBo: apiData.kthcMaCanBo || null,
                    kthcNam: apiData.kthcNam || null,
                    kthcThang: apiData.kthcThang || new Date().getMonth() + 1,
                    kthcplgd: apiData.kthcplgd || '',
                    kthcIdvunglv: apiData.kthcIdvunglv || null,
                    kthcNlb: apiData.kthcNlb || '',
                    kthcPtkt: apiData.kthcPtkt || '',
                    kthcTenCoQuan: localStorage.getItem('kthcTenCoQuan') || '',
                    kthcMSQHNS: localStorage.getItem('kthcMSQHNS') || null,
                    kthcMaDonvi: localStorage.getItem('kthcMaDonvi') || '',
                    currentContext: localStorage.getItem('currentContext') || 'Tạo lương tháng'
                };

                const requestBody = {
                    message: message,
                    sessionId: this.sessionId,
                    isApiCall: isApiCall, // Flag để server biết đây là API call
                    domain: window.location.hostname,
                    userInfo: userInfo,
                    apiMapping: {
                        domain: window.location.hostname,
                        api_server: this.config.apiUrl.replace('/api/chat', ''),
                        environment: window.location.hostname === 'localhost' ? 'development' : 'production'
                    },
                    // Thêm các thông tin API bắt buộc
                    apiParams: {
                        kthcplgd: apiData.kthcplgd,
                        token_httc: tokenWithBearer, // Token với Bearer
                        kthcMaCoQuan: apiData.kthcMaCoQuan,
                        kthcMaCanBo: apiData.kthcMaCanBo,
                        kthcIdvunglv: apiData.kthcIdvunglv,
                        kthcNam: apiData.kthcNam,
                        kthcThang: apiData.kthcThang || new Date().getMonth() + 1,
                        user: apiData.user,
                        currentContext: localStorage.getItem('currentContext') || 'Tạo lương tháng'
                    }
                };
                
                console.log('📤 Gửi request:', requestBody);
                console.log('🔧 isApiCall:', isApiCall);
                console.log('🔑 API Params:', requestBody.apiParams);
                
                const response = await fetch(this.config.apiUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(requestBody)
                });

                const responseText = await response.text();
                console.log('ChatBot Response Text:', responseText);
                
                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (jsonError) {
                    console.error('ChatBot JSON Parse Error:', jsonError);
                    console.error('Response Text:', responseText);
                    this.hideTypingIndicator();
                    this.addMessage(`❌ Lỗi xử lý dữ liệu từ server:\n• Response không phải JSON hợp lệ\n• Status: ${response.status}\n• Text: ${responseText.substring(0, 100)}...`, 'bot');
                    return;
                }

                this.hideTypingIndicator();
                this.hideConnectionStatus();

                if (data.success) {
                    // Kiểm tra nếu có data chi tiết từ API call
                    if (data.data) {
                        const apiData = data.data;
                        let responseMessage = '';
                        
                        // Xử lý response tạo lương/bảo hiểm thành công (không có type)
                        if (apiData.ctgsId && apiData.apiResult) {
                            console.log('📊 === DỮ LIỆU THỰC TỪ API ===');
                            console.log('👥 Total Employees:', apiData.totalEmployees);
                            console.log('💰 Total Salary:', apiData.totalSalary);
                            console.log('📈 Average Salary:', apiData.averageSalary);
                            console.log('🔍 Raw Data:', apiData.rawData);
                            console.log('🏁 === KẾT THÚC LOG ===');
                            
                            // Xác định loại nghiệp vụ dựa vào type hoặc context
                            const currentContext = localStorage.getItem('currentContext') || 'Tạo lương tháng';
                            const isInsurance = currentContext.includes('bảo hiểm');
                            const isUnion = currentContext.includes('công đoàn');
                            
                            let actionText = 'Tạo lương';
                            let buttonText = 'Xem chi tiết lương';
                            let buttonAction = `window.ChatBotAPI.viewSalaryDetails('${apiData.month}', '${apiData.year}')`;
                            
                            if (isInsurance) {
                                actionText = 'Tạo bảo hiểm';
                                buttonText = 'Xem chi tiết bảo hiểm';
                                buttonAction = `window.ChatBotAPI.viewInsuranceDetails('${apiData.month}', '${apiData.year}')`;
                            } else if (isUnion) {
                                actionText = 'Tạo công đoàn';
                                buttonText = 'Xem chi tiết công đoàn';
                                buttonAction = `window.ChatBotAPI.viewUnionDetails('${apiData.month}', '${apiData.year}')`;
                            }
                            
                            // Hiển thị thông tin đơn giản khi thành công
                            responseMessage = `<strong>${actionText} tháng ${apiData.month}/${apiData.year} thành công!</strong>

<div class="chatbot-action-buttons">
    <button class="chatbot-action-btn primary" onclick="${buttonAction}">
        ${buttonText}
    </button>
</div>`;
                            
                            this.addMessage(responseMessage, 'bot');
                        } else if (apiData.type) {
                            // Xử lý các loại response có type
                            switch (apiData.type) {
                                                        case 'ctgs_selection':
                            // Hiển thị danh sách CTGS cho user chọn
                            responseMessage = `**Tìm thấy ${apiData.totalCount} CTGS! Vui lòng chọn một CTGS để tạo lương:**`;
                            
                            // Tạo bảng CTGS
                            const ctgsTable = `
<div class="chatbot-ctgs-table">
    <table>
        <thead>
            <tr>
                <th>STT</th>
                <th>Số hiệu</th>
                <th>Kế toán</th>
                <th>Mã</th>
                <th>Số tiền (VNĐ)</th>
                <th>Tháng</th>
                <th>Thao tác</th>
            </tr>
        </thead>
        <tbody>
            ${apiData.ctgsList.map(ctgs => `
                <tr>
                    <td>${ctgs.index}</td>
                    <td>${ctgs.soHieu}</td>
                    <td>${ctgs.keToan}</td>
                    <td>${ctgs.smtm ? ctgs.smtm.slice(-4) : 'N/A'}</td>
                    <td>${ctgs.dst ? ctgs.dst.toLocaleString('vi-VN') : 'N/A'}</td>
                    <td>${ctgs.smn || 'N/A'}</td>
                    <td>
                        <button class="chatbot-ctgs-select-btn" onclick="window.ChatBotAPI.selectCTGS(${ctgs.index})">
                            Chọn
                        </button>
                    </td>
                </tr>
            `).join('')}
        </tbody>
    </table>
</div>`;
                            
                            responseMessage += ctgsTable;
                            this.addMessage(responseMessage, 'bot');
                                    break;
                            }
                        } else {
                            // Xử lý message thông thường
                            this.addMessage(data.message, 'bot');
                        }
                    } else {
                        // Xử lý message thông thường
                        this.addMessage(data.message, 'bot');
                    }
                    
                    if (this.config.onMessage && typeof this.config.onMessage === 'function') {
                        this.config.onMessage(message, data.message);
                    }
                } else {
                    // Kiểm tra nếu có lỗi không tìm thấy CTGS
                    if (data.error && data.error.includes('Không tìm thấy CTGS nào')) {
                        const errorMessage = data.error + `
                        
<div class="chatbot-action-buttons">
  <button class="chatbot-action-btn primary" onclick="ChatBotAPI.createNewCTGS()">
    📋 Tạo CTGS mới
  </button>
  <button class="chatbot-action-btn secondary" onclick="ChatBotAPI.retryGetCTGS()">
    🔄 Thử lại
  </button>
</div>`;
                        this.addMessage(errorMessage, 'bot');
                    } else {
                        this.addMessage(`❌ Lỗi: ${data.error || 'Không xác định'}`, 'bot');
                    }
                }

            } catch (error) {
                console.error('ChatBot Error sending message:', error);
                this.hideTypingIndicator();
                
                let errorMessage = 'Xin lỗi, không thể kết nối đến server.';
                
                const errorMsg = error.message || '';
                if (error.name === 'TypeError' && errorMsg.includes('fetch')) {
                    errorMessage = '❌ Không thể kết nối đến API server. Vui lòng kiểm tra:\n• URL API có đúng không\n• Server có hoạt động không\n• Kết nối mạng';
                } else if (error.name === 'TypeError' && errorMsg.includes('JSON')) {
                    errorMessage = '❌ Lỗi xử lý dữ liệu từ server';
                } else {
                    errorMessage = `❌ Lỗi: ${errorMsg}`;
                }
                
                this.addMessage(errorMessage, 'bot');
                
                if (this.config.onError && typeof this.config.onError === 'function') {
                    this.config.onError(error);
                }
            }
        }

        debugLocalStorage() {
            console.log('🔍 Debug localStorage:');
            console.log('- chatbot_session_id:', localStorage.getItem('chatbot_session_id'));
            console.log('- Tất cả keys:', Object.keys(localStorage));
            console.log('- Session ID hiện tại:', this.sessionId);
        }

        // Public API methods
        openChat() {
            this.open();
        }

        closeChat() {
            this.close();
        }

        sendChatMessage(message) {
            if (typeof message === 'string' && message.trim()) {
                this.elements.messageInput.value = message;
                this.sendMessage();
            }
        }

        getConfig() {
            return this.config;
        }
    }

    // Initialize ChatBot when DOM is ready
    async function initChatBot() {
        // Load Font Awesome if not already loaded
        if (!document.querySelector('link[href*="font-awesome"]')) {
            const fontAwesome = document.createElement('link');
            fontAwesome.rel = 'stylesheet';
            fontAwesome.href = 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css';
            document.head.appendChild(fontAwesome);
        }

        // Create ChatBot instance
        window.chatbotWidget = new ChatBotWidget();
        
        // Wait for ChatBot to initialize (including API mapping)
        await window.chatbotWidget.init();

        // Expose API methods globally
        window.ChatBotAPI = {
            open: () => window.chatbotWidget.openChat(),
            close: () => window.chatbotWidget.closeChat(),
            sendMessage: (message) => window.chatbotWidget.sendChatMessage(message),
            getConfig: () => window.chatbotWidget.getConfig(),
            debug: () => window.chatbotWidget.debugLocalStorage(),
            setUserData: (userData) => {
                try {
                    if (userData.user) localStorage.setItem('user', userData.user);
                    if (userData.token_httc) localStorage.setItem('token_httc', userData.token_httc);
                    if (userData.kthcMaCoQuan) localStorage.setItem('kthcMaCoQuan', userData.kthcMaCoQuan);
                    if (userData.kthcMaCanBo) localStorage.setItem('kthcMaCanBo', userData.kthcMaCanBo);
                    if (userData.kthcNam) localStorage.setItem('kthcNam', userData.kthcNam);
                    if (userData.kthcThang) localStorage.setItem('kthcThang', userData.kthcThang);
                    if (userData.kthcTenCoQuan) localStorage.setItem('kthcTenCoQuan', userData.kthcTenCoQuan);
                    if (userData.kthcplgd) localStorage.setItem('kthcplgd', userData.kthcplgd);
                    if (userData.kthcIdvunglv) localStorage.setItem('kthcIdvunglv', userData.kthcIdvunglv);
                    if (userData.kthcNlb) localStorage.setItem('kthcNlb', userData.kthcNlb);
                    if (userData.kthcPtkt) localStorage.setItem('kthcPtkt', userData.kthcPtkt);
                    if (userData.kthcMSQHNS) localStorage.setItem('kthcMSQHNS', userData.kthcMSQHNS);
                    if (userData.kthcMaDonvi) localStorage.setItem('kthcMaDonvi', userData.kthcMaDonvi);
                    
                    console.log('✅ Đã set user data:', userData);
                    return true;
                } catch (error) {
                    console.error('❌ Lỗi set user data:', error);
                    return false;
                }
            },
            getSessionId: () => window.chatbotWidget.sessionId,
            saveSession: () => window.chatbotWidget.saveSessionId(),
            loadHistory: () => window.chatbotWidget.loadChatHistory(),
            showSuggestions: () => window.chatbotWidget.showSuggestions(),
            hideSuggestions: () => window.chatbotWidget.hideSuggestions(),
            testSuggestions: () => {
                console.log('🧪 Test suggestions menu');
                console.log('Suggestions button:', window.chatbotWidget.elements.suggestionsBtn);
                console.log('Suggestions menu:', window.chatbotWidget.elements.suggestionsMenu);
                window.chatbotWidget.showSuggestions();
            },
            createNewCTGS: () => {
                console.log('📋 Tạo CTGS mới');
                
                // Disable button để tránh spam click
                const buttons = document.querySelectorAll('.chatbot-action-buttons button');
                buttons.forEach(btn => {
                    btn.disabled = true;
                    btn.style.opacity = '0.6';
                });
                
                window.chatbotWidget.sendChatMessage('Tạo CTGS mới');
            },
                        retryGetCTGS: () => {
                console.log('🔄 Thử lại lấy CTGS');
                
                // Disable button để tránh spam click
                const buttons = document.querySelectorAll('.chatbot-action-buttons button');
                buttons.forEach(btn => {
                    btn.disabled = true;
                    btn.style.opacity = '0.6';
                });
                
                window.ChatBotAPI.createSalaryWithCTGS();
            },
            selectCTGS: (index) => {
                console.log('🎯 User chọn CTGS số:', index);
                
                // Disable tất cả CTGS buttons để tránh spam click
                const ctgsButtons = document.querySelectorAll('.chatbot-ctgs-select-btn');
                ctgsButtons.forEach(btn => {
                    btn.disabled = true;
                    btn.style.opacity = '0.6';
                    btn.textContent = 'Đang xử lý...';
                });
                
                // Gửi request
                                        window.chatbotWidget.guiTinNhanCoCo(index.toString(), true);
            },
            testApiCall: (message) => {
                console.log('🧪 Test API Call:', message);
                window.chatbotWidget.guiTinNhanCoCo(message, true);
            },
            testApiCallDirect: (message) => {
                console.log('🧪 Test API Call Direct:', message);
                // Test trực tiếp với fetch
                fetch(getApiUrl('/api/chat'), {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        message: message,
                        sessionId: window.chatbotWidget.sessionId,
                        isApiCall: true
                    })
                })
                .then(response => response.json())
                .then(data => {
                    console.log('📥 API Response:', data);
                    if (data.success) {
                        window.chatbotWidget.addMessage(data.response, 'bot');
                    }
                })
                .catch(error => {
                    console.error('❌ API Call Error:', error);
                });
            },
            testFunctionAPI: (type, month, year) => {
                console.log('🧪 Test Function API:', type, month, year);
                fetch(getApiUrl(`/api/function/${type}`), {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ month, year })
                })
                .then(response => response.json())
                .then(data => {
                    console.log('📥 Function API Response:', data);
                    if (data.success) {
                        const message = ` **${type.toUpperCase()} API Response:**
                        
📊 **Data:** ${JSON.stringify(data.data, null, 2)}
⏱️ **Processing Time:** ${data.processingTime}ms`;
                        window.chatbotWidget.addMessage(message, 'bot');
                    } else {
                        window.chatbotWidget.addMessage(`❌ Function API Error: ${data.error}`, 'bot');
                    }
                })
                .catch(error => {
                    console.error('❌ Function API Error:', error);
                    window.chatbotWidget.addMessage(`❌ Function API Error: ${error.message}`, 'bot');
                });
            },
            viewFile: (type, month, year) => {
                console.log('📁 Xem file:', type, month, year);
                // Sau này sẽ mở file hoặc chuyển hướng
                const fileName = `${type}_Thang_${month}_${year}.xlsx`;
                const fileUrl = `/files/${fileName}`;
                
                // Mở file trong tab mới
                window.open(fileUrl, '_blank');
                
                // Thông báo cho user
                window.chatbotWidget.addMessage(`📁 Đang mở file: ${fileName}`, 'bot');
            },
            viewDetails: (type) => {
                console.log('📊 Xem chi tiết:', type);
                // Sau này sẽ chuyển hướng đến trang chi tiết
                const detailUrl = `/details/${type}`;
                
                // Mở trang chi tiết trong tab mới
                window.open(detailUrl, '_blank');
                
                // Thông báo cho user
                window.chatbotWidget.addMessage(`📊 Đang mở trang chi tiết: ${type}`, 'bot');
            },
            viewSalaryDetails: (month, year) => {
                console.log('📊 Xem chi tiết lương:', month, year);
                // Sau này sẽ chuyển hướng đến trang chi tiết lương
                const detailUrl = `/salary-details/${month}/${year}`;
                
                // Mở trang chi tiết trong tab mới
                window.open(detailUrl, '_blank');
                
                // Thông báo cho user
                window.chatbotWidget.addMessage(`📊 Đang mở chi tiết lương tháng ${month}/${year}`, 'bot');
            },
            viewInsuranceDetails: (month, year) => {
                console.log('📊 Xem chi tiết bảo hiểm:', month, year);
                // Sau này sẽ chuyển hướng đến trang chi tiết bảo hiểm
                const detailUrl = `/insurance-details/${month}/${year}`;
                
                // Mở trang chi tiết trong tab mới
                window.open(detailUrl, '_blank');
                
                // Thông báo cho user
                window.chatbotWidget.addMessage(`📊 Đang mở chi tiết bảo hiểm tháng ${month}/${year}`, 'bot');
            },
            viewUnionDetails: (month, year) => {
                console.log('📊 Xem chi tiết công đoàn:', month, year);
                // Sau này sẽ chuyển hướng đến trang chi tiết công đoàn
                const detailUrl = `/union-details/${month}/${year}`;
                
                // Mở trang chi tiết trong tab mới
                window.open(detailUrl, '_blank');
                
                // Thông báo cho user
                window.chatbotWidget.addMessage(`📊 Đang mở chi tiết công đoàn tháng ${month}/${year}`, 'bot');
            },
            downloadFile: (type, month, year) => {
                console.log('⬇️ Tải xuống file:', type, month, year);
                // Sau này sẽ tải file
                const fileName = `${type}_Thang_${month}_${year}.xlsx`;
                const downloadUrl = `/download/${fileName}`;
                
                // Tạo link tải xuống
                const link = document.createElement('a');
                link.href = downloadUrl;
                link.download = fileName;
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
                
                // Thông báo cho user
                window.chatbotWidget.addMessage(`⬇️ Đang tải xuống file: ${fileName}`, 'bot');
            },
            forceNewSession: () => {
                window.chatbotWidget.clearSessionId();
                window.chatbotWidget.generateSessionId();
                console.log('🆕 Đã tạo Session ID mới:', window.chatbotWidget.sessionId);
            },
            // API Data Management
            layDuLieuApi: () => {
                try {
                    // Load từng key riêng biệt từ localStorage (không có prefix)
                    const apiData = {
                        user: localStorage.getItem('user'),
                        token_httc: localStorage.getItem('token_httc'),
                        kthcMaDonvi: localStorage.getItem('kthcMaDonvi'),
                        kthcNam: parseInt(localStorage.getItem('kthcNam')),
                        kthcThang: parseInt(localStorage.getItem('kthcThang')),
                        kthcMaCanBo: parseInt(localStorage.getItem('kthcMaCanBo')),
                        kthcTenCoQuan: localStorage.getItem('kthcTenCoQuan'),
                        kthcMSQHNS: parseInt(localStorage.getItem('kthcMSQHNS')),
                        kthcplgd: localStorage.getItem('kthcplgd'),
                        kthcMaCoQuan: parseInt(localStorage.getItem('kthcMaCoQuan')),
                        kthcIdvunglv: parseInt(localStorage.getItem('kthcIdvunglv')),
                        kthcNlb: localStorage.getItem('kthcNlb'),
                        kthcPtkt: localStorage.getItem('kthcPtkt')
                    };
                    
                    // Kiểm tra xem có dữ liệu nào không
                    const hasData = Object.values(apiData).some(value => value !== null && value !== undefined);
                    return hasData ? apiData : null;
                } catch (error) {
                    console.error('❌ Error getting API data:', error);
                    return null;
                }
            },
            luuDuLieuApi: (data) => {
                try {
                    // Save từng key riêng biệt vào localStorage (không có prefix)
                    if (data.user) localStorage.setItem('user', data.user);
                    if (data.token_httc) localStorage.setItem('token_httc', data.token_httc);
                    if (data.kthcMaDonvi) localStorage.setItem('kthcMaDonvi', data.kthcMaDonvi);
                    if (data.kthcNam) localStorage.setItem('kthcNam', data.kthcNam);
                    if (data.kthcThang) localStorage.setItem('kthcThang', data.kthcThang);
                    if (data.kthcMaCanBo) localStorage.setItem('kthcMaCanBo', data.kthcMaCanBo);
                    if (data.kthcTenCoQuan) localStorage.setItem('kthcTenCoQuan', data.kthcTenCoQuan);
                    if (data.kthcMSQHNS) localStorage.setItem('kthcMSQHNS', data.kthcMSQHNS);
                    if (data.kthcplgd) localStorage.setItem('kthcplgd', data.kthcplgd);
                    if (data.kthcMaCoQuan) localStorage.setItem('kthcMaCoQuan', data.kthcMaCoQuan);
                    if (data.kthcIdvunglv) localStorage.setItem('kthcIdvunglv', data.kthcIdvunglv);
                    if (data.kthcNlb) localStorage.setItem('kthcNlb', data.kthcNlb);
                    if (data.kthcPtkt) localStorage.setItem('kthcPtkt', data.kthcPtkt);
                    
                    console.log('💾 API Data set:', data);
                    return true;
                } catch (error) {
                    console.error('❌ Error setting API data:', error);
                    return false;
                }
            },
            sendMessageWithApiData: (message) => {
                try {
                    const apiData = window.ChatBotAPI.layDuLieuApi();
                    if (apiData) {
                        const enhancedMessage = `${message}\n\n📊 API Data:\n- User: ${apiData.user}\n- Đơn vị: ${apiData.kthcMaDonvi}\n- Năm: ${apiData.kthcNam}\n- Tháng: ${apiData.kthcThang || new Date().getMonth() + 1}\n- Cán bộ: ${apiData.kthcMaCanBo}\n- Cơ quan: ${apiData.kthcTenCoQuan}`;
                        window.chatbotWidget.sendChatMessage(enhancedMessage);
                    } else {
                        window.chatbotWidget.sendChatMessage(message);
                    }
                } catch (error) {
                    console.error('❌ Error sending message with API data:', error);
                    window.chatbotWidget.sendChatMessage(message);
                }
            },
            // CTGS API Methods
            getIDCTGSMax: async (maCoQuan, maCanBo, nam, plgd) => {
                try {
                    const apiData = window.ChatBotAPI.layDuLieuApi();
                    if (!apiData) {
                        throw new Error('Chưa có dữ liệu API');
                    }
                    
                    // Use provided parameters or fallback to API data
                    const params = new URLSearchParams({
                        MaCoQuan: maCoQuan || apiData.kthcMaCoQuan,
                        MaCanBo: maCanBo || apiData.kthcMaCanBo,
                        Nam: nam || apiData.kthcNam,
                        PLGD: plgd || apiData.kthcplgd
                    });
                    
                    const response = await fetch(getApiUrl(`/api/ctgs/getidctgsmax?${params}`), {
                        method: 'GET',
                        headers: {
                            'Content-Type': 'application/json',
                        }
                    });
                    
                    const data = await response.json();
                    return data;
                } catch (error) {
                    console.error('❌ Error getting IDCTGS max:', error);
                    throw error;
                }
            },
            getCTGSList: async () => {
                try {
                    const apiData = window.ChatBotAPI.layDuLieuApi();
                    if (!apiData) {
                        throw new Error('Chưa có dữ liệu API');
                    }
                    
                    // Tự động thêm "Bearer" vào token nếu chưa có
                    const tokenWithBearer = apiData.token_httc.startsWith('Bearer ') 
                        ? apiData.token_httc 
                        : `Bearer ${apiData.token_httc}`;
                    
                    const response = await fetch(getApiUrl('/api/ctgs/list'), {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            kthcplgd: apiData.kthcplgd,
                            token_httc: tokenWithBearer,
                            kthcMaCoQuan: apiData.kthcMaCoQuan,
                            kthcMaCanBo: apiData.kthcMaCanBo,
                            kthcIdvunglv: apiData.kthcIdvunglv
                        })
                    });
                    
                    const data = await response.json();
                    return data;
                } catch (error) {
                    console.error('❌ Error getting CTGS list:', error);
                    throw error;
                }
            },
            createWithCTGS: async (type, ctgsId, month, year) => {
                try {
                    const apiData = window.ChatBotAPI.layDuLieuApi();
                    if (!apiData) {
                        throw new Error('Chưa có dữ liệu API');
                    }
                    
                    // Tự động thêm "Bearer" vào token nếu chưa có
                    const tokenWithBearer = apiData.token_httc.startsWith('Bearer ') 
                        ? apiData.token_httc 
                        : `Bearer ${apiData.token_httc}`;
                    
                    // Gọi API thực tế thay vì mock
                    const response = await fetch(getApiUrl('/api/chat'), {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            sessionId: window.ChatBotAPI.getSessionId(),
                            message: type === 'salary' ? 'Tạo lương tháng' : 
                                    type === 'insurance' ? 'Tạo bảo hiểm tháng' : 
                                    'Tạo kinh phí công đoàn tháng',
                            isApiCall: true,
                            apiParams: {
                                user: apiData.user || 'User',
                                token_httc: apiData.token_httc,
                                kthcplgd: apiData.kthcplgd,
                                kthcMaCoQuan: apiData.kthcMaCoQuan,
                                kthcMaCanBo: apiData.kthcMaCanBo,
                                kthcIdvunglv: apiData.kthcIdvunglv,
                                kthcNam: year,
                                kthcThang: month
                            }
                        })
                    });
                    
                    const data = await response.json();
                    
                    // Chuyển đổi response format để tương thích
                    if (data.success) {
                        return {
                            success: true,
                            data: {
                                type: type,
                                ctgsId: ctgsId,
                                month: month,
                                year: year,
                                message: data.message,
                                responseTime: 0, // Sẽ được tính từ server
                                statusCode: 200
                            }
                        };
                    } else {
                        return {
                            success: false,
                            error: data.error || 'Lỗi không xác định'
                        };
                    }
                } catch (error) {
                    console.error(`❌ Error creating ${type}:`, error);
                    throw error;
                }
            },
            // Enhanced API methods with CTGS workflow
            // Helper function to get API parameters from localStorage
            getApiParams: () => {
                try {
                    // Lấy dữ liệu từ localStorage (không có prefix như setApiData)
                    const user = localStorage.getItem('user');
                    const token_httc = localStorage.getItem('token_httc');
                    const kthcplgd = localStorage.getItem('kthcplgd');
                    const kthcMaCoQuan = localStorage.getItem('kthcMaCoQuan');
                    const kthcMaCanBo = localStorage.getItem('kthcMaCanBo');
                    const kthcIdvunglv = localStorage.getItem('kthcIdvunglv');
                    const kthcNam = localStorage.getItem('kthcNam');
                    const kthcNlb = localStorage.getItem('kthcNlb');
                    const kthcPtkt = localStorage.getItem('kthcPtkt');
                    
                    // Kiểm tra dữ liệu bắt buộc
                    const missingData = [];
                    if (!user) missingData.push('user');
                    if (!token_httc) missingData.push('token_httc');
                    if (!kthcMaCoQuan) missingData.push('kthcMaCoQuan');
                    if (!kthcMaCanBo) missingData.push('kthcMaCanBo');
                    if (!kthcIdvunglv) missingData.push('kthcIdvunglv');
                    if (!kthcNam) missingData.push('kthcNam');
                    
                    if (missingData.length > 0) {
                        throw new Error(`Lỗi truyền tham số. Chỉ sử dụng ChatBot sau khi đã đăng nhập KTHCOnline`);
                    }
                    
                    const apiParams = {
                        user: user,
                        token_httc: token_httc,
                        kthcplgd: kthcplgd || 'KH-CB',
                        kthcMaCoQuan: parseInt(kthcMaCoQuan),
                        kthcMaCanBo: parseInt(kthcMaCanBo),
                        kthcIdvunglv: parseInt(kthcIdvunglv),
                        kthcNam: parseInt(kthcNam),
                        kthcNlb: kthcNlb || '',
                        kthcPtkt: kthcPtkt || ''
                    };
                    
                    console.log('🔍 === API PARAMS TỪ LOCALSTORAGE ===');
                    console.log('📋 API Params:', apiParams);
                    console.log('🏁 === KẾT THÚC LOG ===');
                    
                    return apiParams;
                } catch (error) {
                    console.error('❌ Lỗi khi lấy API params từ localStorage:', error);
                    throw error; // Ném lỗi thay vì trả về giá trị mặc định
                }
            },
            
            // Hàm chung để tạo lương/bảo hiểm/công đoàn với CTGS
            createWithCTGS: async (type) => {
                try {
                    // Lấy API params từ localStorage
                    const apiParams = window.ChatBotAPI.getApiParams();
                    
                    // Xác định message dựa trên type
                    let message;
                    let logTitle;
                    let errorMessage;
                    
                    switch (type) {
                        case 'salary':
                            message = 'Tạo lương tháng';
                            logTitle = 'TẠO LƯƠNG';
                            errorMessage = 'Lỗi không xác định khi tạo lương';
                            break;
                        case 'insurance':
                            message = 'Tạo bảo hiểm tháng';
                            logTitle = 'TẠO BẢO HIỂM';
                            errorMessage = 'Lỗi không xác định khi tạo bảo hiểm';
                            break;
                        case 'union':
                            message = 'Tạo kinh phí công đoàn tháng';
                            logTitle = 'TẠO CÔNG ĐOÀN';
                            errorMessage = 'Lỗi không xác định khi tạo công đoàn';
                            break;
                        default:
                            throw new Error(`Loại không được hỗ trợ: ${type}`);
                    }
                    
                    // Gọi trực tiếp API
                    const response = await fetch(getApiUrl('/api/chat'), {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            sessionId: window.chatbotWidget.sessionId,
                            message: message,
                            isApiCall: true,
                            apiParams: apiParams
                        })
                    });
                    
                    // Kiểm tra response status
                    if (!response.ok) {
                        const errorText = await response.text();
                        console.error(`❌ HTTP Error ${response.status}:`, errorText);
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    
                    // Kiểm tra content type
                    const contentType = response.headers.get('content-type');
                    if (!contentType || !contentType.includes('application/json')) {
                        const errorText = await response.text();
                        console.error('❌ Invalid content type:', contentType);
                        console.error('❌ Response text:', errorText.substring(0, 200));
                        throw new Error('Server trả về không phải JSON');
                    }
                    
                    const result = await response.json();
                    
                    // Log kết quả chi tiết
                    console.log(`📊 === KẾT QUẢ ${logTitle} ===`);
                    console.log(' Thành công:', result.success);
                    console.log('📋 Dữ liệu:', result.data);
                    console.log('🏁 === KẾT THÚC LOG ===\n');
                    
                    return result;
                } catch (error) {
                    console.error(`❌ Error in createWithCTGS (${type}):`, error);
                    
                    // Kiểm tra nếu là lỗi network
                    if (error.name === 'TypeError' && error.message.includes('fetch')) {
                        return {
                            success: false,
                            error: 'Không thể kết nối đến server. Vui lòng kiểm tra kết nối mạng.',
                            details: {
                                message: error.message,
                                type: 'NETWORK_ERROR'
                            }
                        };
                    }
                    
                    // Trả về lỗi có thông tin chi tiết
                    return {
                        success: false,
                        error: error.message || errorMessage,
                        details: {
                            message: error.message,
                            type: 'API_PARAMS_ERROR'
                        }
                    };
                }
            },
            
            // Wrapper functions để giữ tương thích
            createSalaryWithCTGS: async () => {
                return await window.ChatBotAPI.createWithCTGS('salary');
            },
            
            createInsuranceWithCTGS: async () => {
                return await window.ChatBotAPI.createWithCTGS('insurance');
            },
            

            createUnionWithCTGS: async () => {
                return await window.ChatBotAPI.createWithCTGS('union');
            },
            
            // Test connection method
            testConnection: async () => {
                try {
                    console.log('🧪 Testing connection to server...');
                    
                    const response = await fetch(getApiUrl('/api/test'), {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            test: true,
                            timestamp: new Date().toISOString()
                        })
                    });
                    
                    if (!response.ok) {
                        const errorText = await response.text();
                        console.error(`❌ Test failed: HTTP ${response.status}`);
                        console.error('Response:', errorText);
                        return {
                            success: false,
                            error: `HTTP ${response.status}: ${response.statusText}`
                        };
                    }
                    
                    const result = await response.json();
                    console.log('✅ Test successful:', result);
                    return result;
                    
                } catch (error) {
                    console.error('❌ Test connection error:', error);
                    return {
                        success: false,
                        error: error.message
                    };
                }
            },
            
            // Error handling methods
            retryCreateSalary: async () => {
                try {
                    const apiData = window.ChatBotAPI.getApiData();
                    if (!apiData) {
                        throw new Error('Chưa có dữ liệu API');
                    }
                    
                    console.log('🔄 Đang thử lại tạo lương...');
                    
                    const response = await fetch(getApiUrl('/api/retry-create-salary'), {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            token: apiData.token_httc,
                            kthcMaCoQuan: apiData.kthcMaCoQuan,
                            kthcMaCanBo: apiData.kthcMaCanBo,
                            kthcNam: apiData.kthcNam || new Date().getFullYear(),
                            kthcThang: apiData.kthcThang || new Date().getMonth() + 1
                        })
                    });
                    
                    const result = await response.json();
                    
                    if (result.success && result.retryResult.success) {
                        console.log(' Thử lại thành công!');
                        return {
                            success: true,
                            message: 'Thử lại tạo lương thành công!',
                            data: result.retryResult
                        };
                    } else {
                        console.log('❌ Thử lại thất bại:', result.retryResult?.message);
                        return {
                            success: false,
                            error: result.retryResult?.message || 'Thử lại thất bại'
                        };
                    }
                } catch (error) {
                    console.error('❌ Error retrying create salary:', error);
                    throw error;
                }
            },
            
            checkDataStatus: async () => {
                try {
                    const apiData = window.ChatBotAPI.getApiData();
                    if (!apiData) {
                        throw new Error('Chưa có dữ liệu API');
                    }
                    
                    console.log('🔍 Đang kiểm tra trạng thái dữ liệu...');
                    
                    const response = await fetch(getApiUrl('/api/check-data-status'), {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            token: apiData.token_httc,
                            kthcMaCoQuan: apiData.kthcMaCoQuan,
                            kthcMaCanBo: apiData.kthcMaCanBo
                        })
                    });
                    
                    const result = await response.json();
                    
                    if (result.success) {
                        console.log('📊 Trạng thái dữ liệu:', result.dataStatus);
                        return {
                            success: true,
                            message: `Có ${result.dataStatus.employeeCount} nhân viên trong hệ thống`,
                            data: result.dataStatus
                        };
                    } else {
                        console.log('❌ Kiểm tra dữ liệu thất bại:', result.error);
                        return {
                            success: false,
                            error: result.error || 'Kiểm tra dữ liệu thất bại'
                        };
                    }
                } catch (error) {
                    console.error('❌ Error checking data status:', error);
                    throw error;
                }
            },
            
            contactSupport: () => {
                console.log('📞 Liên hệ hỗ trợ...');
                alert('Vui lòng liên hệ admin để được hỗ trợ:\n- Email: admin@company.com\n- Phone: 0123-456-789');
            }
        };
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initChatBot);
    } else {
        initChatBot();
    }

})();
