// Dashboard JavaScript với Ant Design
class Dashboard {
    constructor() {
        this.currentTab = 'overview';
        this.currentPage = 1;
        this.itemsPerPage = 10;
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadData();
    }

    bindEvents() {
        // Navigation
        const navLinks = document.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const target = e.currentTarget.getAttribute('data-tab');
                this.switchTab(target);
            });
        });

        // Search functionality
        const searchInput = document.getElementById('searchSession');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => {
                this.searchSessions(e.target.value);
            });
        }
    }

    switchTab(tabName) {
        // Update navigation
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });
        document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');

        // Update content
        document.querySelectorAll('.tab-pane').forEach(pane => {
            pane.classList.remove('active');
        });
        document.getElementById(tabName).classList.add('active');

        this.currentTab = tabName;
        this.currentPage = 1; // Reset về trang 1 khi chuyển tab
        this.loadTabData(tabName);
    }

    async loadData() {
        this.showLoading();
        try {
            // Kiểm tra kết nối database trước
            try {
                const healthResponse = await fetch('/api/statistics/health');
                const healthData = await healthResponse.json();
                console.log('Database health:', healthData);
                
                if (!healthData.success || healthData.data.database === 'disconnected') {
                    this.showToast('Cảnh báo: Kết nối database có vấn đề', 'warning');
                }
            } catch (healthError) {
                console.warn('Health check failed:', healthError);
            }
            
            await Promise.all([
                this.loadOverviewData(),
                this.loadApiMappings(),
                this.loadQADatabase(),
                this.loadStatistics(),
                this.loadChatSessions()
            ]);
        } catch (error) {
            this.showToast('Lỗi khi tải dữ liệu: ' + error.message, 'error');
        } finally {
            this.hideLoading();
        }
    }

    async loadTabData(tabName) {
        switch (tabName) {
            case 'overview':
                await this.loadOverviewData();
                break;
            case 'api-mapping':
                await this.loadApiMappings();
                break;
            case 'qa-database':
                await this.loadQADatabase();
                break;
            case 'statistics':
                await this.loadStatistics();
                break;
            case 'chat-history':
                await this.loadChatSessions();
                break;
        }
    }

    async loadOverviewData() {
        try {
            const response = await fetch('/api/statistics/overview');
            const data = await response.json();
            
            if (data.success) {
                // Cập nhật số liệu với format đẹp hơn
                const formatNumber = (num) => {
                    if (num >= 1000000) {
                        return (num / 1000000).toFixed(1) + 'M';
                    } else if (num >= 1000) {
                        return (num / 1000).toFixed(1) + 'K';
                    }
                    return num.toLocaleString();
                };

                document.getElementById('totalSessions').textContent = formatNumber(data.data.totalSessions || 0);
                document.getElementById('totalMessages').textContent = formatNumber(data.data.totalMessages || 0);
                document.getElementById('apiCalls').textContent = formatNumber(data.data.apiCalls || 0);
                document.getElementById('qaResponses').textContent = formatNumber(data.data.qaResponses || 0);
                
                // Thêm thông tin về ChatGPT responses và lỗi nếu có element
                const chatgptElement = document.getElementById('chatgptResponses');
                if (chatgptElement) {
                    chatgptElement.textContent = formatNumber(data.data.chatgptResponses || 0);
                }
                
                const errorsElement = document.getElementById('errors');
                if (errorsElement) {
                    errorsElement.textContent = formatNumber(data.data.errors || 0);
                }

                // Cập nhật tooltip với thông tin chi tiết hơn
                this.updateStatsTooltips(data.data);
                
                // Cập nhật thông tin bổ sung
                this.updateAdditionalStats(data.data);
                
                this.renderCharts(data.data);
            }
        } catch (error) {
            console.error('Error loading overview data:', error);
        }
    }

    updateStatsTooltips(data) {
        const tooltips = {
            'totalSessions': `Tổng số phiên chat: ${data.totalSessions?.toLocaleString() || 0}\n\nMỗi session đại diện cho một cuộc hội thoại riêng biệt giữa người dùng và chatbot. Session được tạo khi người dùng bắt đầu chat và kết thúc khi người dùng rời khỏi hoặc timeout.`,
            'totalMessages': `Tổng số tin nhắn: ${data.totalMessages?.toLocaleString() || 0}\n\nBao gồm tất cả tin nhắn đã trao đổi trong hệ thống:\n• Tin nhắn từ người dùng\n• Phản hồi từ chatbot\n• Tin nhắn hệ thống\n• Thông báo lỗi`,
            'apiCalls': `Số lần gọi API: ${data.apiCalls?.toLocaleString() || 0}\n\nCác API được gọi bao gồm:\n• KTHC API (thông tin học viên)\n• Lương API (thông tin lương)\n• Bảo hiểm API (thông tin bảo hiểm)\n• Công đoàn API (thông tin công đoàn)`,
            'qaResponses': `Phản hồi từ Q&A DB: ${data.qaResponses?.toLocaleString() || 0}\n\nHệ thống trả lời dựa trên cơ sở dữ liệu Q&A có sẵn thay vì sử dụng ChatGPT. Điều này giúp:\n• Tăng tốc độ phản hồi\n• Giảm chi phí API\n• Đảm bảo độ chính xác`,
            'chatgptResponses': `Phản hồi từ ChatGPT: ${data.chatgptResponses?.toLocaleString() || 0}\n\nSố lần sử dụng ChatGPT API để tạo phản hồi thông minh. ChatGPT được sử dụng khi:\n• Không có câu trả lời trong Q&A DB\n• Câu hỏi phức tạp cần xử lý AI\n• Yêu cầu tạo nội dung mới`,
            'errors': `Số lỗi hệ thống: ${data.errors?.toLocaleString() || 0}\n\nCác loại lỗi bao gồm:\n• Lỗi kết nối API\n• Lỗi database\n• Lỗi xử lý tin nhắn\n• Lỗi timeout\n• Lỗi validation dữ liệu`
        };

        // Cập nhật tooltip cho từng card
        Object.keys(tooltips).forEach(id => {
            const element = document.getElementById(id);
            if (element) {
                const card = element.closest('.stats-card');
                if (card) {
                    card.setAttribute('title', tooltips[id]);
                }
            }
        });
    }

    updateAdditionalStats(data) {
        // Cập nhật thời gian phản hồi trung bình
        const avgResponseTimeElement = document.getElementById('avgResponseTime');
        if (avgResponseTimeElement) {
            const avgTime = data.avgResponseTime || 0;
            avgResponseTimeElement.textContent = avgTime.toLocaleString();
        }

        // Tính tỷ lệ thành công
        const successRateElement = document.getElementById('successRate');
        if (successRateElement) {
            const totalMessages = data.totalMessages || 0;
            const errors = data.errors || 0;
            const successRate = totalMessages > 0 ? ((totalMessages - errors) / totalMessages * 100).toFixed(1) : 100;
            successRateElement.textContent = successRate;
            
            // Thêm màu sắc dựa trên tỷ lệ thành công
            const successRateNum = parseFloat(successRate);
            if (successRateNum >= 95) {
                successRateElement.style.color = '#10b981'; // Xanh lá
            } else if (successRateNum >= 90) {
                successRateElement.style.color = '#f59e0b'; // Vàng
            } else {
                successRateElement.style.color = '#ef4444'; // Đỏ
            }
        }

        // Cập nhật trạng thái database
        const dbStatusElement = document.getElementById('dbStatus');
        if (dbStatusElement) {
            // Kiểm tra kết nối database
            fetch('/api/statistics/health')
                .then(response => response.json())
                .then(healthData => {
                    if (healthData.success && healthData.data.database === 'connected') {
                        dbStatusElement.textContent = 'Online';
                        dbStatusElement.style.color = '#10b981';
                    } else {
                        dbStatusElement.textContent = 'Offline';
                        dbStatusElement.style.color = '#ef4444';
                    }
                })
                .catch(() => {
                    dbStatusElement.textContent = 'Lỗi';
                    dbStatusElement.style.color = '#ef4444';
                });
        }
    }

    async loadApiMappings() {
        try {
            const response = await fetch('/api/statistics/api-mappings');
            const data = await response.json();
            
            if (data.success) {
                this.renderApiMappingsTable(data.data);
            }
        } catch (error) {
            console.error('Error loading API mappings:', error);
        }
    }

    async loadQADatabase() {
        try {
            const response = await fetch('/api/statistics/qa-database');
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            
            console.log('QA Database response:', data); // Debug log
            
            if (data.success && data.data) {
                // Lưu dữ liệu gốc vào cache để sử dụng cho filtering
                this.originalQAData = data.data;
                console.log('QA Data loaded:', data.data.length, 'items'); // Debug log
                this.renderQADatabaseTable(data.data);
            } else {
                console.warn('API trả về dữ liệu không hợp lệ:', data);
                this.originalQAData = [];
                this.renderQADatabaseTable([]);
            }
        } catch (error) {
            console.error('Error loading QA database:', error);
            // Hiển thị thông báo lỗi cho người dùng
            this.originalQAData = [];
            this.renderQADatabaseTable([]);
        }
    }

    async loadStatistics() {
        try {
            const response = await fetch('/api/statistics/overview');
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.success && data.data) {
                document.getElementById('avgResponseTime').textContent = data.data.avgResponseTime || 0;
                
                // Tính số người dùng hoạt động (sessions unique trong 7 ngày gần nhất)
                try {
                    const activeUsersResponse = await fetch('/api/statistics/daily');
                    if (activeUsersResponse.ok) {
                        const activeUsersData = await activeUsersResponse.json();
                        let activeUsers = 0;
                        if (activeUsersData.success && activeUsersData.data && activeUsersData.data.stats) {
                            activeUsers = activeUsersData.data.stats.reduce((sum, stat) => sum + stat.total_sessions, 0);
                        }
                        document.getElementById('activeUsers').textContent = activeUsers;
                    }
                } catch (activeUsersError) {
                    console.error('Error loading active users:', activeUsersError);
                    document.getElementById('activeUsers').textContent = '0';
                }
                
                await this.renderDailyStatistics(data.data);
            } else {
                console.warn('API trả về dữ liệu không hợp lệ:', data);
                document.getElementById('avgResponseTime').textContent = '0';
                document.getElementById('activeUsers').textContent = '0';
            }
        } catch (error) {
            console.error('Error loading statistics:', error);
            document.getElementById('avgResponseTime').textContent = '0';
            document.getElementById('activeUsers').textContent = '0';
        }
    }

    async loadChatSessions() {
        try {
            const response = await fetch('/api/statistics/chat-sessions');
            const data = await response.json();
            
            if (data.success) {
                // Lưu dữ liệu gốc để sử dụng cho delete functions
                this.originalSessionsData = data.data;
                this.renderChatSessionsTable(data.data);
            }
        } catch (error) {
            console.error('Error loading chat sessions:', error);
        }
    }

    renderApiMappingsTable(mappings) {
        const container = document.getElementById('apiMappingTable');
        
        if (!mappings || mappings.length === 0) {
            container.innerHTML = `
                <div style="text-align: center; padding: 40px; color: var(--text-muted);">
                    <i class="fas fa-database" style="font-size: 3rem; margin-bottom: 16px; opacity: 0.5;"></i>
                    <p>Chưa có API mapping nào</p>
                    <button class="btn btn-primary" onclick="showAddMappingModal()">
                        <i class="fas fa-plus"></i> Thêm mapping đầu tiên
                    </button>
                </div>
            `;
            return;
        }

        // Phân trang
        const totalPages = Math.ceil(mappings.length / this.itemsPerPage);
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const paginatedMappings = mappings.slice(startIndex, endIndex);

        let html = `
            <div style="background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border); overflow: hidden;">
                <table class="table" style="margin: 0;">
                    <thead>
                        <tr style="background: #f8fafc;">
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-globe"></i> Tên Miền
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-server"></i> Máy Chủ API
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-layer-group"></i> Môi Trường
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-info-circle"></i> Mô Tả
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-toggle-on"></i> Trạng Thái
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-cogs"></i> Thao Tác
                            </th>
                        </tr>
                    </thead>
                    <tbody>
        `;

        paginatedMappings.forEach(mapping => {
            const statusClass = mapping.is_active ? 'tag-development' : 'tag-production';
            const statusText = mapping.is_active ? 'Hoạt động' : 'Đã tắt';
            
            html += `
                <tr style="border-bottom: 1px solid var(--border);">
                    <td style="padding: 12px 16px;">
                        <div style="display: flex; flex-direction: column; gap: 4px;">
                            <div style="font-weight: 500; color: var(--text-primary); font-size: 14px;">
                                <i class="fas fa-globe" style="color: var(--primary-color); margin-right: 6px;"></i>
                                ${mapping.domain}
                            </div>
                            <div style="font-size: 11px; color: var(--text-muted);">
                                <i class="fas fa-clock"></i>
                                Tạo: ${this.formatDate(mapping.created_at)}
                            </div>
                        </div>
                    </td>
                    <td style="padding: 12px 16px;">
                        <div style="display: flex; flex-direction: column; gap: 4px;">
                            <div style="font-weight: 500; color: var(--text-primary); font-size: 13px; word-break: break-all;">
                                <i class="fas fa-server" style="color: var(--primary-color); margin-right: 6px;"></i>
                                ${mapping.api_server}
                            </div>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span style="width: 8px; height: 8px; background: #10b981; border-radius: 50%; display: inline-block;"></span>
                                <span style="font-size: 11px; color: var(--text-muted);">Trực tuyến</span>
                            </div>
                        </div>
                    </td>
                    <td style="padding: 12px 16px;">
                        <span class="tag tag-${mapping.environment}" style="font-size: 12px; padding: 6px 12px;">
                            <i class="fas ${this.getEnvironmentIcon(mapping.environment)}" style="margin-right: 4px;"></i>
                            ${this.getEnvironmentText(mapping.environment)}
                        </span>
                    </td>
                    <td style="padding: 12px 16px;">
                        <div style="max-width: 200px;">
                            ${mapping.description ? 
                                `<div style="font-size: 13px; color: var(--text-primary); line-height: 1.4;">${mapping.description}</div>` : 
                                `<div style="font-size: 12px; color: var(--text-muted); font-style: italic;">Không có mô tả</div>`
                            }
                        </div>
                    </td>
                    <td style="padding: 12px 16px;">
                        <span class="tag ${statusClass}" style="font-size: 12px; padding: 6px 12px;">
                            <i class="fas fa-circle" style="font-size: 8px; margin-right: 4px;"></i>
                            ${statusText}
                        </span>
                    </td>
                    <td style="padding: 12px 16px;">
                        <div style="display: flex; gap: 6px;">
                            <button class="btn btn-sm btn-secondary" onclick="editMapping(${mapping.id})" style="font-size: 11px; padding: 6px 10px;" title="Sửa">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-primary" onclick="testMapping('${mapping.api_server}')" style="font-size: 11px; padding: 6px 10px;" title="Kiểm tra">
                                <i class="fas fa-play"></i>
                            </button>
                            <button class="btn btn-sm btn-danger" onclick="deleteMapping(${mapping.id})" style="font-size: 11px; padding: 6px 10px;" title="Xóa">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        });

        html += `
                    </tbody>
                </table>
            </div>
        `;

        // Thêm phân trang
        if (totalPages > 1) {
            html += this.renderPagination(totalPages, 'api-mapping');
        }

        container.innerHTML = html;
    }

    renderQADatabaseTable(qaData) {
        const container = document.getElementById('qaDatabaseTable');
        
        console.log('renderQADatabaseTable called with:', qaData); // Debug log
        
        if (!qaData || qaData.length === 0) {
            console.log('No QA data to display'); // Debug log
            container.innerHTML = `
                <div style="text-align: center; padding: 40px; color: var(--text-muted);">
                    <i class="fas fa-question-circle" style="font-size: 3rem; margin-bottom: 16px; opacity: 0.5;"></i>
                    <p>Chưa có Q&A nào</p>
                    <button class="btn btn-primary" onclick="showAddQAModal()">
                        <i class="fas fa-plus"></i> Thêm Q&A đầu tiên
                    </button>
                </div>
            `;
            return;
        }

        // Sử dụng dữ liệu gốc trực tiếp (không cần lọc ở đây vì đây là lần load đầu tiên)
        const groupedQAData = qaData;

        console.log('Grouped QA data:', groupedQAData.length, 'items'); // Debug log

        // Phân trang
        const totalPages = Math.ceil(groupedQAData.length / this.itemsPerPage);
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const paginatedQAData = groupedQAData.slice(startIndex, endIndex);
        
        console.log('Pagination:', { totalPages, startIndex, endIndex, paginatedQAData: paginatedQAData.length }); // Debug log

        // Tạo controls tìm kiếm và lọc
        const searchControls = `
            <div style="background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border); padding: 16px; margin-bottom: 16px;">
                <div style="display: flex; align-items: center; gap: 16px; flex-wrap: wrap;">
                    <div style="display: flex; align-items: center; gap: 8px; flex: 1; min-width: 200px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Tìm kiếm:</label>
                        <input type="text" id="qaSearchInput" onkeyup="filterQAData()" placeholder="Tìm kiếm câu hỏi hoặc câu trả lời..." 
                               style="flex: 1; padding: 8px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px;">
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Danh mục:</label>
                        <select id="qaCategoryFilter" onchange="filterQAData()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 120px;">
                            <option value="">Tất cả danh mục</option>
                            ${this.getUniqueCategories(qaData).map(category => `<option value="${category}">${category}</option>`).join('')}
                        </select>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Trạng thái:</label>
                        <select id="qaStatusFilter" onchange="filterQAData()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 100px;">
                            <option value="">Tất cả</option>
                            <option value="active">Hoạt động</option>
                            <option value="inactive">Đã tắt</option>
                        </select>
                    </div>
                    
                    <button onclick="clearQAFilters()" class="btn btn-sm btn-secondary" style="font-size: 12px;">
                        <i class="fas fa-times"></i> Xóa lọc
                    </button>
                </div>
            </div>
        `;

        let html = searchControls + `
            <div style="background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border); overflow: hidden;">
                <table class="table" style="margin: 0;">
                    <thead>
                        <tr style="background: #f8fafc;">
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-folder"></i> Danh Mục
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-question"></i> Câu Hỏi
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-comment"></i> Câu Trả Lời
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-star"></i> Điểm Chính Xác
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-toggle-on"></i> Trạng Thái
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-cogs"></i> Thao Tác
                            </th>
                        </tr>
                    </thead>
                    <tbody>
        `;

        paginatedQAData.forEach(qa => {
            const statusClass = qa.is_active ? 'tag-development' : 'tag-production';
            const statusText = qa.is_active ? 'Hoạt động' : 'Đã tắt';
            
            html += `
                <tr>
                    <td>
                        <span class="tag tag-info">
                            <i class="fas fa-folder"></i>
                            ${qa.category || 'Không phân loại'}
                        </span>
                    </td>
                    <td>
                        <div class="question-cell">
                            <div class="question-text">${qa.question}</div>
                        </div>
                    </td>
                    <td>
                        <div class="answer-cell">
                            <div class="answer-text">${qa.answer.substring(0, 100)}${qa.answer.length > 100 ? '...' : ''}</div>
                        </div>
                    </td>
                    <td>
                        <div class="score-cell">
                            <span class="score-badge">${qa.score || 0}/10</span>
                        </div>
                    </td>
                    <td>
                        <span class="tag ${statusClass}">
                            <i class="fas fa-check"></i>
                            ${statusText}
                        </span>
                    </td>
                    <td>
                        <div class="action-buttons">
                            <button class="btn btn-sm btn-secondary" onclick="editQA(${qa.id})">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-danger" onclick="deleteQA(${qa.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        });

        html += `
                </tbody>
            </table>
        `;

        // Thêm phân trang
        if (totalPages > 1) {
            html += this.renderPagination(totalPages, 'qa-database');
        }

        container.innerHTML = html;
    }

    renderChatSessionsTable(sessions) {
        const container = document.getElementById('chatSessionsTable');
        
        if (!sessions || sessions.length === 0) {
            container.innerHTML = `
                <div style="text-align: center; padding: 40px; color: var(--text-muted);">
                    <i class="fas fa-comments" style="font-size: 3rem; margin-bottom: 16px; opacity: 0.5;"></i>
                    <p>Chưa có chat session nào</p>
                </div>
            `;
            return;
        }

        // Tạo combo lọc thông minh
        const domains = [...new Set(sessions.map(s => {
            const userInfo = s.user_info ? (typeof s.user_info === 'string' ? JSON.parse(s.user_info) : s.user_info) : {};
            return userInfo.domain || 'Unknown';
        }))];
        
        const users = [...new Set(sessions.map(s => {
            const userInfo = s.user_info ? (typeof s.user_info === 'string' ? JSON.parse(s.user_info) : s.user_info) : {};
            return userInfo.user || 'Anonymous';
        }))];

        // Phân trang cho sessions đã nhóm
        const groupedSessions = this.groupSessionsByDomainAndUser(sessions);
        const totalPages = Math.ceil(groupedSessions.length / this.itemsPerPage);
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const paginatedGroupedSessions = groupedSessions.slice(startIndex, endIndex);

        const filterControls = `
            <div style="margin-bottom: 20px; padding: 16px; background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border);">
                <div style="display: flex; align-items: center; gap: 16px; flex-wrap: wrap;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-weight: 500; color: var(--text-primary); font-size: 14px;">Lọc theo:</label>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Tên miền:</label>
                        <select id="domainFilter" onchange="filterSessions()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 120px;">
                            <option value="">Tất cả</option>
                            ${domains.map(domain => `<option value="${domain}">${domain}</option>`).join('')}
                        </select>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Người dùng:</label>
                        <select id="userFilter" onchange="filterSessions()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 120px;">
                            <option value="">Tất cả</option>
                            ${users.map(user => `<option value="${user}">${user}</option>`).join('')}
                        </select>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Trạng thái:</label>
                        <select id="statusFilter" onchange="filterSessions()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 100px;">
                            <option value="">Tất cả</option>
                            <option value="active">Hoạt động</option>
                            <option value="inactive">Đã đóng</option>
                        </select>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Từ ngày:</label>
                        <input type="date" id="dateFromFilter" onchange="filterSessions()" lang="vi-VN" placeholder="dd/mm/yyyy" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 120px;">
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Đến ngày:</label>
                        <input type="date" id="dateToFilter" onchange="filterSessions()" lang="vi-VN" placeholder="dd/mm/yyyy" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 120px;">
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <button onclick="setDateFilter('today')" class="btn btn-sm btn-primary" style="font-size: 11px;">
                            <i class="fas fa-calendar-day"></i> Hôm nay
                        </button>
                        <button onclick="setDateFilter('week')" class="btn btn-sm btn-primary" style="font-size: 11px;">
                            <i class="fas fa-calendar-week"></i> Tuần này
                        </button>
                        <button onclick="setDateFilter('month')" class="btn btn-sm btn-primary" style="font-size: 11px;">
                            <i class="fas fa-calendar-alt"></i> Tháng này
                        </button>
                    </div>
                    
                    <button onclick="clearFilters()" class="btn btn-sm btn-secondary" style="font-size: 12px;">
                        <i class="fas fa-times"></i> Xóa lọc
                    </button>
                </div>
            </div>
        `;

        let html = filterControls + `
            <div style="background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border); overflow: hidden;">
                <table class="table" style="margin: 0;">
                    <thead>
                        <tr style="background: #f8fafc;">
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-globe"></i> Tên Miền
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-user"></i> Người Dùng
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-hashtag"></i> Sessions
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-comments"></i> Tin Nhắn
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-clock"></i> Hoạt Động Cuối
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-toggle-on"></i> Trạng Thái
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-cogs"></i> Thao Tác
                            </th>
                        </tr>
                    </thead>
                    <tbody>
        `;

        // Nhóm sessions theo domain và user
        const groupedSessionsData = {};
        sessions.forEach(session => {
            const userInfo = session.user_info ? (typeof session.user_info === 'string' ? JSON.parse(session.user_info) : session.user_info) : {};
            const domain = userInfo.domain || 'Unknown';
            const user = userInfo.user || 'Anonymous';
            
            if (!groupedSessionsData[domain]) {
                groupedSessionsData[domain] = {};
            }
            
            if (!groupedSessionsData[domain][user]) {
                groupedSessionsData[domain][user] = {
                    userInfo: userInfo,
                    sessions: [],
                    totalMessages: 0,
                    lastActivity: null,
                    activeSessions: 0
                };
            }
            
            groupedSessionsData[domain][user].sessions.push(session);
            groupedSessionsData[domain][user].totalMessages += session.message_count || 0;
            
            if (session.is_active) {
                groupedSessionsData[domain][user].activeSessions++;
            }
            
            if (!groupedSessionsData[domain][user].lastActivity || session.last_activity > groupedSessionsData[domain][user].lastActivity) {
                groupedSessionsData[domain][user].lastActivity = session.last_activity;
            }
        });

        // Render sessions theo domain và user (đã phân trang)
        paginatedGroupedSessions.forEach(({ domain, user, userData }) => {
                const statusClass = userData.activeSessions > 0 ? 'tag-development' : 'tag-production';
                const statusText = userData.activeSessions > 0 ? 'Hoạt động' : 'Đã đóng';
                
                html += `
                    <tr class="session-row" data-domain="${domain}" data-user="${user}" data-status="${userData.activeSessions > 0 ? 'active' : 'inactive'}" data-last-activity="${userData.lastActivity}">
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <div style="display: flex; align-items: center; gap: 8px;">
                                <i class="fas fa-globe" style="color: var(--primary-color); font-size: 14px;"></i>
                                <span style="font-weight: 500; color: var(--text-primary);">${domain}</span>
                            </div>
                        </td>
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <div style="display: flex; flex-direction: column; gap: 4px;">
                                <span style="font-weight: 500; color: var(--text-primary);">${user}</span>
                                ${userData.userInfo.kthcTenCoQuan ? `<small style="color: var(--text-muted); font-size: 11px;"><i class="fas fa-building"></i> ${userData.userInfo.kthcTenCoQuan}</small>` : ''}
                                ${userData.userInfo.kthcMaCanBo ? `<small style="color: var(--text-muted); font-size: 11px;">Mã CB: ${userData.userInfo.kthcMaCanBo}</small>` : ''}
                            </div>
                        </td>
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span class="group-badge blue" style="width: 20px; height: 20px; font-size: 11px;">${userData.sessions.length}</span>
                                <span style="font-size: 12px; color: var(--text-secondary);">sessions</span>
                            </div>
                        </td>
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <span style="font-weight: 600; color: var(--text-primary);">${userData.totalMessages}</span>
                            <br><small style="color: var(--text-muted); font-size: 11px;">tin nhắn</small>
                        </td>
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <span style="font-size: 12px; color: var(--text-primary);">${this.formatDate(userData.lastActivity)}</span>
                        </td>
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <span class="tag ${statusClass}" style="font-size: 11px;">
                                <i class="fas fa-circle" style="font-size: 8px;"></i>
                                ${statusText}
                            </span>
                        </td>
                        <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                            <div style="display: flex; gap: 6px;">
                                <button class="btn btn-sm btn-primary" onclick="viewUserSessions('${domain}', '${user}')" style="font-size: 11px; padding: 4px 8px;">
                                    <i class="fas fa-eye"></i> Xem
                                </button>
                                <button class="btn btn-sm btn-secondary" onclick="expandUserSessions('${domain}', '${user}')" style="font-size: 11px; padding: 4px 8px;">
                                    <i class="fas fa-expand"></i> Chi tiết
                                </button>
                                <button class="btn btn-sm btn-danger" onclick="deleteUserSessions('${domain}', '${user}')" style="font-size: 11px; padding: 4px 8px;">
                                    <i class="fas fa-trash"></i> Xóa
                                </button>
                            </div>
                        </td>
                    </tr>
                `;
                
                // Thêm chi tiết sessions (ẩn mặc định)
                userData.sessions.forEach(session => {
                    const sessionStatusClass = session.is_active ? 'tag-development' : 'tag-production';
                    const sessionStatusText = session.is_active ? 'Hoạt động' : 'Đã đóng';
                    
                    html += `
                        <tr class="session-detail" data-domain="${domain}" data-user="${user}" style="display: none; background-color: #f8fafc;">
                            <td style="padding: 12px 16px 12px 32px; border-bottom: 1px solid var(--border);">
                                <small style="color: var(--text-muted); font-size: 11px;">${session.session_id}</small>
                            </td>
                            <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                                <small style="color: var(--text-muted); font-size: 11px;">${user}</small>
                            </td>
                            <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                                <span style="font-size: 11px; color: var(--text-secondary);">1 session</span>
                            </td>
                            <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                                <span style="font-size: 11px; color: var(--text-primary);">${session.message_count || 0}</span>
                            </td>
                            <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                                <span style="font-size: 11px; color: var(--text-secondary);">${this.formatDate(session.last_activity)}</span>
                            </td>
                            <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                                <span class="tag ${sessionStatusClass}" style="font-size: 10px;">
                                    <i class="fas fa-circle" style="font-size: 6px;"></i>
                                    ${sessionStatusText}
                                </span>
                            </td>
                            <td style="padding: 12px 16px; border-bottom: 1px solid var(--border);">
                                <div style="display: flex; gap: 4px;">
                                    <button class="btn btn-sm btn-primary" onclick="viewChatHistory('${session.session_id}')" style="font-size: 10px; padding: 3px 6px;">
                                        <i class="fas fa-eye"></i> Xem
                                    </button>
                                    <button class="btn btn-sm btn-danger" onclick="deleteSession('${session.session_id}')" style="font-size: 10px; padding: 3px 6px;">
                                        <i class="fas fa-trash"></i> Xóa
                                    </button>
                                </div>
                            </td>
                        </tr>
                    `;
                });
        });

        html += `
                    </tbody>
                </table>
            </div>
        `;

        // Thêm phân trang
        if (totalPages > 1) {
            html += this.renderPagination(totalPages, 'chat-history');
        }

        container.innerHTML = html;
    }

    async renderDailyStatistics(data) {
        // Render daily statistics chart với dữ liệu thực tế
        try {
            const response = await fetch('/api/statistics/daily');
            const chartData = await response.json();
            
            const ctx = document.getElementById('dailyStatsChart');
            if (ctx && chartData.success && chartData.data.stats) {
                // Destroy existing chart if it exists
                if (this.dailyStatsChart) {
                    this.dailyStatsChart.destroy();
                }
                
                this.dailyStatsChart = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: chartData.data.stats.map(stat => {
                            const date = new Date(stat.date);
                            return date.toLocaleDateString('vi-VN', { day: '2-digit', month: '2-digit' });
                        }),
                        datasets: [{
                            label: 'Messages',
                            data: chartData.data.stats.map(stat => stat.total_messages),
                            borderColor: '#6366f1',
                            backgroundColor: 'rgba(99, 102, 241, 0.1)',
                            tension: 0.4
                        }, {
                            label: 'API Calls',
                            data: chartData.data.stats.map(stat => stat.api_calls),
                            borderColor: '#f59e0b',
                            backgroundColor: 'rgba(245, 158, 11, 0.1)',
                            tension: 0.4
                        }, {
                            label: 'Q&A Responses',
                            data: chartData.data.stats.map(stat => stat.qa_responses),
                            borderColor: '#10b981',
                            backgroundColor: 'rgba(16, 185, 129, 0.1)',
                            tension: 0.4
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'top',
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            }
        } catch (error) {
            console.error('Error rendering daily statistics:', error);
        }
    }

    renderCharts(data) {
        // Response type chart
        const responseCtx = document.getElementById('responseChart');
        if (responseCtx) {
            // Destroy existing chart if it exists
            if (this.responseChart) {
                this.responseChart.destroy();
            }
            
            this.responseChart = new Chart(responseCtx, {
                type: 'doughnut',
                data: {
                    labels: ['API Calls', 'Q&A Responses', 'API Chatbot', 'Lỗi'],
                    datasets: [{
                        data: [data.apiCalls || 0, data.qaResponses || 0, data.chatgptResponses || 0, data.errors || 0],
                        backgroundColor: ['#6366f1', '#10b981', '#4f46e5', '#ef4444'],
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                        }
                    }
                }
            });
        }

        // Activity chart - sử dụng dữ liệu thực tế từ API
        this.loadActivityChart();
    }

    async loadActivityChart() {
        try {
            const response = await fetch('/api/statistics/daily');
            const data = await response.json();
            
            if (data.success && data.data.stats) {
                const activityCtx = document.getElementById('activityChart');
                if (activityCtx) {
                    // Destroy existing chart if it exists
                    if (this.activityChart) {
                        this.activityChart.destroy();
                    }
                    
                    this.activityChart = new Chart(activityCtx, {
                        type: 'line',
                        data: {
                            labels: data.data.stats.map(stat => {
                                const date = new Date(stat.date);
                                return date.toLocaleDateString('vi-VN', { day: '2-digit', month: '2-digit' });
                            }),
                            datasets: [{
                                label: 'Messages',
                                data: data.data.stats.map(stat => stat.total_messages),
                                borderColor: '#6366f1',
                                backgroundColor: 'rgba(99, 102, 241, 0.1)',
                                tension: 0.4
                            }, {
                                label: 'Sessions',
                                data: data.data.stats.map(stat => stat.total_sessions),
                                borderColor: '#10b981',
                                backgroundColor: 'rgba(16, 185, 129, 0.1)',
                                tension: 0.4
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'top',
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true
                                }
                            }
                        }
                    });
                }
            }
        } catch (error) {
            console.error('Error loading activity chart:', error);
        }
    }

    renderChatHistoryModal(sessionId, messages) {
        const content = document.getElementById('chatHistoryContent');
        content.innerHTML = '';
        document.getElementById('chatSessionId').textContent = sessionId;

        messages.forEach(message => {
            const messageDiv = document.createElement('div');
            messageDiv.className = `chat-message ${message.role}`;

            const avatar = document.createElement('div');
            avatar.className = `chat-avatar ${message.role}`;
            avatar.innerHTML = message.role === 'user' ? '<i class="fas fa-user"></i>' : '<i class="fas fa-robot"></i>';

            const messageContent = document.createElement('div');
            messageContent.className = 'chat-content';

            messageContent.innerHTML = `
                <div class="chat-header">
                    <div class="chat-sender">${message.role === 'user' ? 'User' : 'Bot'}</div>
                    <div class="chat-time">${this.formatDate(message.created_at)}</div>
                </div>
                <div class="chat-text">${message.content}</div>
            `;

            messageDiv.appendChild(avatar);
            messageDiv.appendChild(messageContent);
            content.appendChild(messageDiv);
        });

        // Show modal
        document.getElementById('chatHistoryModal').classList.add('show');
    }

    searchSessions(query) {
        const rows = document.querySelectorAll('#chatSessionsTable tbody tr');
        rows.forEach(row => {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(query.toLowerCase()) ? '' : 'none';
        });
    }

    showLoading() {
        document.getElementById('loading').classList.add('show');
    }

    hideLoading() {
        document.getElementById('loading').classList.remove('show');
    }

    showToast(message, type = 'success', title = null) {
        const container = document.getElementById('toastContainer');
        const toast = document.createElement('div');
        toast.className = 'toast';
        
        const iconClass = type === 'success' ? 'success' : 
                         type === 'error' ? 'error' : 
                         type === 'warning' ? 'warning' : 'info';
        
        const icon = type === 'success' ? 'check' : 
                    type === 'error' ? 'exclamation-triangle' : 
                    type === 'warning' ? 'exclamation-circle' : 'info-circle';
        
        const toastTitle = title || (type === 'success' ? 'Thành công' : 
                                   type === 'error' ? 'Lỗi' : 
                                   type === 'warning' ? 'Cảnh báo' : 'Thông báo');
        
        toast.innerHTML = `
            <div class="toast-icon ${iconClass}">
                <i class="fas fa-${icon}"></i>
            </div>
            <div class="toast-content">
                <div class="toast-title">${toastTitle}</div>
                <div class="toast-message">${message}</div>
            </div>
            <button class="toast-close" onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        `;
        
        container.appendChild(toast);
        
        // Hiển thị toast
        setTimeout(() => {
            toast.classList.add('show');
        }, 100);
        
        // Tự động ẩn sau 5 giây
        setTimeout(() => {
            toast.classList.add('hide');
            setTimeout(() => {
                if (toast.parentElement) {
                    toast.remove();
                }
            }, 300);
        }, 5000);
    }

    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('vi-VN', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    getEnvironmentIcon(environment) {
        switch (environment) {
            case 'production': return 'fa-rocket';
            case 'development': return 'fa-code';
            case 'demo': return 'fa-flask';
            default: return 'fa-layer-group';
        }
    }

    getEnvironmentText(environment) {
        switch (environment) {
            case 'production': return 'Sản xuất';
            case 'development': return 'Phát triển';
            case 'demo': return 'Demo';
            default: return environment;
        }
    }

    // Helper methods for time-based grouping
    getUniqueUsers(sessions) {
        const users = new Set();
        sessions.forEach(session => {
            const userInfo = session.user_info ? (typeof session.user_info === 'string' ? JSON.parse(session.user_info) : session.user_info) : {};
            users.add(userInfo.user || 'Anonymous');
        });
        return Array.from(users);
    }

    getLatestActivity(sessions) {
        if (sessions.length === 0) return null;
        return sessions.reduce((latest, session) => {
            const sessionDate = new Date(session.last_activity);
            const latestDate = latest ? new Date(latest) : new Date(0);
            return sessionDate > latestDate ? session.last_activity : latest;
        }, null);
    }

    // Function để nhóm sessions theo domain và user
    groupSessionsByDomainAndUser(sessions) {
        const grouped = {};
        sessions.forEach(session => {
            const userInfo = session.user_info ? (typeof session.user_info === 'string' ? JSON.parse(session.user_info) : session.user_info) : {};
            const domain = userInfo.domain || 'Unknown';
            const user = userInfo.user || 'Anonymous';
            
            if (!grouped[domain]) {
                grouped[domain] = {};
            }
            
            if (!grouped[domain][user]) {
                grouped[domain][user] = {
                    userInfo: userInfo,
                    sessions: [],
                    totalMessages: 0,
                    lastActivity: null,
                    activeSessions: 0
                };
            }
            
            grouped[domain][user].sessions.push(session);
            grouped[domain][user].totalMessages += session.message_count || 0;
            
            if (session.is_active) {
                grouped[domain][user].activeSessions++;
            }
            
            if (!grouped[domain][user].lastActivity || session.last_activity > grouped[domain][user].lastActivity) {
                grouped[domain][user].lastActivity = session.last_activity;
            }
        });

        // Chuyển đổi thành array để phân trang
        const result = [];
        Object.entries(grouped).forEach(([domain, users]) => {
            Object.entries(users).forEach(([user, userData]) => {
                result.push({
                    domain: domain,
                    user: user,
                    userData: userData
                });
            });
        });

        return result;
    }

    // Function để render phân trang
    renderPagination(totalPages, tabName) {
        let html = `
            <div class="pagination-container">
                <div style="display: flex; align-items: center; gap: 4px;">
                    <span style="font-size: 12px; color: var(--text-secondary);">Trang:</span>
        `;

        // Nút Previous
        if (this.currentPage > 1) {
            html += `
                <button onclick="Dashboard.getInstance().changePage(${this.currentPage - 1}, '${tabName}')" class="btn btn-sm btn-secondary" style="font-size: 11px; padding: 4px 8px;">
                    <i class="fas fa-chevron-left"></i>
                </button>
            `;
        }

        // Các nút trang
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);

        for (let i = startPage; i <= endPage; i++) {
            const isActive = i === this.currentPage;
            html += `
                <button onclick="Dashboard.getInstance().changePage(${i}, '${tabName}')" 
                        class="btn btn-sm ${isActive ? 'btn-primary' : 'btn-secondary'}" 
                        style="font-size: 11px; padding: 4px 8px; min-width: 32px;">
                    ${i}
                </button>
            `;
        }

        // Nút Next
        if (this.currentPage < totalPages) {
            html += `
                <button onclick="Dashboard.getInstance().changePage(${this.currentPage + 1}, '${tabName}')" class="btn btn-sm btn-secondary" style="font-size: 11px; padding: 4px 8px;">
                    <i class="fas fa-chevron-right"></i>
                </button>
            `;
        }

        html += `
                </div>
                <div style="margin-left: 16px;">
                    <span style="font-size: 12px; color: var(--text-secondary);">
                        Trang ${this.currentPage} / ${totalPages}
                    </span>
                </div>
            </div>
        `;

        return html;
    }

    // Function để thay đổi trang
    changePage(page, tabName) {
        this.currentPage = page;
        
        // Xử lý đặc biệt cho Q&A Database với filtering
        if (tabName === 'qa-database' && this.originalQAData) {
            this.filterQAData();
        } else {
            this.loadTabData(tabName);
        }
    }

    // API Methods
    async loadMappingData(id) {
        try {
            const response = await fetch(`/api/statistics/api-mappings/${id}`);
            const data = await response.json();
            
            if (data.success) {
                const mapping = data.data;
                document.getElementById('mappingId').value = mapping.id;
                document.getElementById('mappingDomain').value = mapping.domain;
                document.getElementById('mappingEnvironment').value = mapping.environment;
                document.getElementById('mappingApiServer').value = mapping.api_server;
                document.getElementById('mappingDescription').value = mapping.description || '';
                document.getElementById('mappingActive').checked = mapping.is_active;
                
                document.getElementById('mappingModalTitle').textContent = 'Sửa API Mapping';
            }
        } catch (error) {
            this.showToast('Lỗi khi tải dữ liệu mapping: ' + error.message, 'error');
        }
    }

    async saveMapping() {
        const formData = {
            domain: document.getElementById('mappingDomain').value,
            environment: document.getElementById('mappingEnvironment').value,
            api_server: document.getElementById('mappingApiServer').value,
            description: document.getElementById('mappingDescription').value,
            is_active: document.getElementById('mappingActive').checked
        };

        const mappingId = document.getElementById('mappingId').value;

        try {
            const url = mappingId ? `/api/statistics/api-mappings/${mappingId}` : '/api/statistics/api-mappings';
            const method = mappingId ? 'PUT' : 'POST';
            
            console.log('Saving mapping:', { url, method, formData });
            
            const response = await fetch(url, {
                method: method,
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(formData)
            });

            console.log('Response status:', response.status);
            
            if (!response.ok) {
                const errorText = await response.text();
                console.error('API Error:', errorText);
                throw new Error(`HTTP ${response.status}: ${errorText}`);
            }

            const data = await response.json();
            console.log('Response data:', data);
            
            if (data.success) {
                this.showToast('Lưu API mapping thành công!');
                closeMappingModal();
                this.loadApiMappings();
            } else {
                this.showToast('Lỗi: ' + (data.error || 'Không xác định'), 'error');
            }
        } catch (error) {
            console.error('Save mapping error:', error);
            this.showToast('Lỗi khi lưu mapping: ' + error.message, 'error');
        }
    }

    async deleteMapping(id) {
        if (!confirm('Bạn có chắc chắn muốn xóa API mapping này?')) return;

        try {
            const response = await fetch(`/api/statistics/api-mappings/${id}`, {
                method: 'DELETE'
            });

            const data = await response.json();
            
            if (data.success) {
                this.showToast('Xóa API mapping thành công!');
                this.loadApiMappings();
            } else {
                this.showToast('Lỗi: ' + data.error, 'error');
            }
        } catch (error) {
            this.showToast('Lỗi khi xóa mapping: ' + error.message, 'error');
        }
    }

    async loadQAData(id) {
        try {
            const response = await fetch(`/api/statistics/qa-database/${id}`);
            const data = await response.json();
            
            if (data.success) {
                const qa = data.data;
                document.getElementById('qaId').value = qa.id;
                document.getElementById('qaQuestion').value = qa.question;
                document.getElementById('qaAnswer').value = qa.answer;
                document.getElementById('qaCategory').value = qa.category || '';
                document.getElementById('qaScore').value = qa.score || 0;
                document.getElementById('qaKeywords').value = qa.keywords || '';
                document.getElementById('qaActive').checked = qa.is_active;
                
                document.getElementById('qaModalTitle').textContent = 'Sửa Q&A';
            }
        } catch (error) {
            this.showToast('Lỗi khi tải dữ liệu Q&A: ' + error.message, 'error');
        }
    }

    async saveQA() {
        const formData = {
            id: document.getElementById('qaId').value,
            question: document.getElementById('qaQuestion').value,
            answer: document.getElementById('qaAnswer').value,
            category: document.getElementById('qaCategory').value,
            score: document.getElementById('qaScore').value,
            keywords: document.getElementById('qaKeywords').value,
            is_active: document.getElementById('qaActive').checked
        };

        try {
            const url = formData.id ? `/api/statistics/qa-database/${formData.id}` : '/api/statistics/qa-database';
            const method = formData.id ? 'PUT' : 'POST';
            
            const response = await fetch(url, {
                method: method,
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(formData)
            });

            const data = await response.json();
            
            if (data.success) {
                this.showToast('Lưu Q&A thành công!');
                closeQAModal();
                this.loadQADatabase();
            } else {
                this.showToast('Lỗi: ' + data.error, 'error');
            }
        } catch (error) {
            this.showToast('Lỗi khi lưu Q&A: ' + error.message, 'error');
        }
    }

    async deleteQA(id) {
        if (!confirm('Bạn có chắc chắn muốn xóa Q&A này?')) return;

        try {
            const response = await fetch(`/api/statistics/qa-database/${id}`, {
                method: 'DELETE'
            });

            const data = await response.json();
            
            if (data.success) {
                this.showToast('Xóa Q&A thành công!');
                this.loadQADatabase();
            } else {
                this.showToast('Lỗi: ' + data.error, 'error');
            }
        } catch (error) {
            this.showToast('Lỗi khi xóa Q&A: ' + error.message, 'error');
        }
    }

    // ==================== Q&A Database Helper Methods ====================

    // Lọc dữ liệu Q&A
    filterQAData() {
        const searchTerm = document.getElementById('qaSearchInput')?.value?.toLowerCase() || '';
        const categoryFilter = document.getElementById('qaCategoryFilter')?.value || '';
        const statusFilter = document.getElementById('qaStatusFilter')?.value || '';

        // Lấy dữ liệu gốc từ cache hoặc reload
        if (!this.originalQAData) {
            this.loadQADatabase();
            return;
        }

        let filteredData = this.originalQAData.filter(qa => {
            // Lọc theo từ khóa tìm kiếm
            const matchesSearch = !searchTerm || 
                qa.question.toLowerCase().includes(searchTerm) ||
                qa.answer.toLowerCase().includes(searchTerm);

            // Lọc theo danh mục
            const matchesCategory = !categoryFilter || qa.category === categoryFilter;

            // Lọc theo trạng thái
            const matchesStatus = !statusFilter || 
                (statusFilter === 'active' && qa.is_active) ||
                (statusFilter === 'inactive' && !qa.is_active);

            return matchesSearch && matchesCategory && matchesStatus;
        });

        // Nhóm theo category và render
        const groupedData = this.groupQADataByCategory(filteredData);
        this.renderQADatabaseTableWithData(groupedData);
    }

    // Nhóm dữ liệu Q&A theo category
    groupQADataByCategory(qaData) {
        // Kiểm tra qaData có tồn tại và là array không
        if (!qaData || !Array.isArray(qaData)) {
            console.warn('qaData không hợp lệ:', qaData);
            return [];
        }
        
        const grouped = {};
        
        qaData.forEach(qa => {
            const category = qa.category || 'Không phân loại';
            if (!grouped[category]) {
                grouped[category] = [];
            }
            grouped[category].push(qa);
        });

        // Chuyển đổi thành array và sắp xếp theo category
        const result = [];
        Object.keys(grouped).sort().forEach(category => {
            grouped[category].forEach(qa => {
                result.push(qa);
            });
        });

        return result;
    }

    // Lấy danh sách unique categories
    getUniqueCategories(qaData) {
        // Kiểm tra qaData có tồn tại và là array không
        if (!qaData || !Array.isArray(qaData)) {
            console.warn('qaData không hợp lệ trong getUniqueCategories:', qaData);
            return [];
        }
        
        const categories = new Set();
        qaData.forEach(qa => {
            if (qa && qa.category) {
                categories.add(qa.category);
            }
        });
        return Array.from(categories).sort();
    }

        // Render Q&A table với dữ liệu đã được xử lý
    renderQADatabaseTableWithData(groupedQAData) {
        const container = document.getElementById('qaDatabaseTable');
        
        // Lưu trạng thái focus và cursor position của search input
        const searchInput = document.getElementById('qaSearchInput');
        const wasFocused = searchInput && document.activeElement === searchInput;
        const cursorPosition = wasFocused ? searchInput.selectionStart : -1;
        
        // Tạo controls tìm kiếm và lọc (luôn hiển thị)
        const searchControls = `
            <div style="background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border); padding: 16px; margin-bottom: 16px;">
                <div style="display: flex; align-items: center; gap: 16px; flex-wrap: wrap;">
                    <div style="display: flex; align-items: center; gap: 8px; flex: 1; min-width: 200px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Tìm kiếm:</label>
                        <input type="text" id="qaSearchInput" onkeyup="filterQAData()" placeholder="Tìm kiếm câu hỏi hoặc câu trả lời..." 
                               value="${document.getElementById('qaSearchInput')?.value || ''}"
                               style="flex: 1; padding: 8px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px;">
                    </div>

                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Danh mục:</label>
                        <select id="qaCategoryFilter" onchange="filterQAData()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 120px;">
                            <option value="">Tất cả danh mục</option>
                            ${this.getUniqueCategories(this.originalQAData || []).map(category =>
                                `<option value="${category}" ${document.getElementById('qaCategoryFilter')?.value === category ? 'selected' : ''}>${category}</option>`
                            ).join('')}
                        </select>
                    </div>

                    <div style="display: flex; align-items: center; gap: 8px;">
                        <label style="font-size: 12px; color: var(--text-secondary);">Trạng thái:</label>
                        <select id="qaStatusFilter" onchange="filterQAData()" style="padding: 6px 12px; border: 1px solid var(--border); border-radius: 4px; font-size: 12px; min-width: 100px;">
                            <option value="">Tất cả</option>
                            <option value="active" ${document.getElementById('qaStatusFilter')?.value === 'active' ? 'selected' : ''}>Đang hoạt động</option>
                            <option value="inactive" ${document.getElementById('qaStatusFilter')?.value === 'inactive' ? 'selected' : ''}>Đã tắt</option>
                        </select>
                    </div>

                    <button onclick="clearQAFilters()" class="btn btn-sm btn-secondary" style="font-size: 12px;">
                        <i class="fas fa-times"></i> Xóa lọc
                    </button>
                </div>
            </div>
        `;

        if (!groupedQAData || groupedQAData.length === 0) {
            container.innerHTML = searchControls + `
                <div style="text-align: center; padding: 40px; color: var(--text-muted);">
                    <i class="fas fa-question-circle" style="font-size: 3rem; margin-bottom: 16px; opacity: 0.5;"></i>
                    <p>Không tìm thấy Q&A nào phù hợp</p>
                </div>
            `;
            
            // Khôi phục focus và cursor position nếu cần
            if (wasFocused) {
                const newSearchInput = document.getElementById('qaSearchInput');
                if (newSearchInput) {
                    newSearchInput.focus();
                    if (cursorPosition >= 0) {
                        newSearchInput.setSelectionRange(cursorPosition, cursorPosition);
                    }
                }
            }
            return;
        }

        // Phân trang
        const totalPages = Math.ceil(groupedQAData.length / this.itemsPerPage);
        const startIndex = (this.currentPage - 1) * this.itemsPerPage;
        const endIndex = startIndex + this.itemsPerPage;
        const paginatedQAData = groupedQAData.slice(startIndex, endIndex);

        let html = searchControls + `
            <div style="background: var(--surface); border-radius: var(--radius); border: 1px solid var(--border); overflow: hidden;">
                <table class="table" style="margin: 0;">
                    <thead>
                        <tr style="background: #f8fafc;">
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-folder"></i> Danh Mục
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-question"></i> Câu Hỏi
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-comment"></i> Câu Trả Lời
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-star"></i> Điểm Chính Xác
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-toggle-on"></i> Trạng Thái
                            </th>
                            <th style="padding: 12px 16px; font-size: 13px; font-weight: 600; color: var(--text-secondary);">
                                <i class="fas fa-cogs"></i> Thao Tác
                            </th>
                        </tr>
                    </thead>
                    <tbody>
        `;

        paginatedQAData.forEach(qa => {
            const statusClass = qa.is_active ? 'tag-development' : 'tag-production';
            const statusText = qa.is_active ? 'Hoạt động' : 'Đã tắt';
            
            html += `
                <tr>
                    <td>
                        <span class="tag tag-info">
                            <i class="fas fa-folder"></i>
                            ${qa.category || 'Không phân loại'}
                        </span>
                    </td>
                    <td>
                        <div class="question-cell">
                            <div class="question-text">${qa.question}</div>
                        </div>
                    </td>
                    <td>
                        <div class="answer-cell">
                            <div class="answer-text">${qa.answer.substring(0, 100)}${qa.answer.length > 100 ? '...' : ''}</div>
                        </div>
                    </td>
                    <td>
                        <div class="score-cell">
                            <span class="score-badge">${qa.score || 0}/10</span>
                        </div>
                    </td>
                    <td>
                        <span class="tag ${statusClass}">
                            <i class="fas fa-check"></i>
                            ${statusText}
                        </span>
                    </td>
                    <td>
                        <div class="action-buttons">
                            <button class="btn btn-sm btn-secondary" onclick="editQA(${qa.id})">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-danger" onclick="deleteQA(${qa.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        });

        html += `
                </tbody>
            </table>
        `;

        // Thêm phân trang
        if (totalPages > 1) {
            html += this.renderPagination(totalPages, 'qa-database');
        }

        container.innerHTML = html;
        
        // Khôi phục focus và cursor position nếu cần
        if (wasFocused) {
            const newSearchInput = document.getElementById('qaSearchInput');
            if (newSearchInput) {
                newSearchInput.focus();
                if (cursorPosition >= 0) {
                    newSearchInput.setSelectionRange(cursorPosition, cursorPosition);
                }
            }
        }
    }
}

// Global functions for modal management
function showAddMappingModal() {
    document.getElementById('mappingModalTitle').textContent = 'Thêm API Mapping';
    document.getElementById('mappingForm').reset();
    document.getElementById('mappingId').value = '';
    document.getElementById('mappingModal').classList.add('show');
}

function closeMappingModal() {
    document.getElementById('mappingModal').classList.remove('show');
}

function editMapping(id) {
    Dashboard.getInstance().loadMappingData(id);
    document.getElementById('mappingModal').classList.add('show');
}

function deleteMapping(id) {
    Dashboard.getInstance().deleteMapping(id);
}

function saveMapping() {
    Dashboard.getInstance().saveMapping();
}

function showAddQAModal() {
    document.getElementById('qaModalTitle').textContent = 'Thêm Q&A';
    document.getElementById('qaForm').reset();
    document.getElementById('qaId').value = '';
    document.getElementById('qaModal').classList.add('show');
}

function closeQAModal() {
    document.getElementById('qaModal').classList.remove('show');
}

function editQA(id) {
    Dashboard.getInstance().loadQAData(id);
    document.getElementById('qaModal').classList.add('show');
}

function deleteQA(id) {
    Dashboard.getInstance().deleteQA(id);
}

function saveQA() {
    Dashboard.getInstance().saveQA();
}

function closeChatHistoryModal() {
    document.getElementById('chatHistoryModal').classList.remove('show');
}

function viewChatHistory(sessionId) {
    fetch(`/api/statistics/chat-history/${sessionId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                Dashboard.getInstance().renderChatHistoryModal(sessionId, data.data.messages);
            } else {
                Dashboard.getInstance().showToast('Lỗi: ' + data.error, 'error');
            }
        })
        .catch(error => {
            Dashboard.getInstance().showToast('Lỗi khi tải lịch sử chat: ' + error.message, 'error');
        });
}

function testMapping(apiServer) {
    Dashboard.getInstance().showToast(`Đang test kết nối đến ${apiServer}...`, 'success');
    // Implement actual API testing logic here
}

// Function để lọc sessions
function filterSessions() {
    const domainFilter = document.getElementById('domainFilter').value;
    const userFilter = document.getElementById('userFilter').value;
    const statusFilter = document.getElementById('statusFilter').value;
    const dateFromFilter = document.getElementById('dateFromFilter').value;
    const dateToFilter = document.getElementById('dateToFilter').value;
    
    const rows = document.querySelectorAll('.session-row');
    
    rows.forEach(row => {
        const domain = row.dataset.domain;
        const user = row.dataset.user;
        const status = row.dataset.status;
        const lastActivity = row.dataset.lastActivity;
        
        let show = true;
        
        // Lọc theo domain
        if (domainFilter && domain !== domainFilter) {
            show = false;
        }
        
        // Lọc theo user
        if (userFilter && user !== userFilter) {
            show = false;
        }
        
        // Lọc theo status
        if (statusFilter && status !== statusFilter) {
            show = false;
        }
        
        // Lọc theo ngày - Sửa timezone issue
        if (lastActivity && (dateFromFilter || dateToFilter)) {
            try {
                const activityDate = new Date(lastActivity);
                
                // Đảm bảo activityDate là hợp lệ
                if (isNaN(activityDate.getTime())) {
                    console.warn('Invalid date:', lastActivity);
                    show = false;
                } else {
                    // Sử dụng local date thay vì UTC để tránh timezone issue
                    const year = activityDate.getFullYear();
                    const month = String(activityDate.getMonth() + 1).padStart(2, '0');
                    const day = String(activityDate.getDate()).padStart(2, '0');
                    const activityDateStr = `${year}-${month}-${day}`;
                    
                    // Debug log để kiểm tra format ngày
                    console.log('Debug date filter for row:', {
                        domain: domain,
                        user: user,
                        lastActivity: lastActivity,
                        activityDate: activityDate,
                        activityDateStr: activityDateStr,
                        dateFromFilter: dateFromFilter,
                        dateToFilter: dateToFilter,
                        fromCondition: dateFromFilter ? activityDateStr >= dateFromFilter : true,
                        toCondition: dateToFilter ? activityDateStr <= dateToFilter : true
                    });
                    
                    if (dateFromFilter && activityDateStr < dateFromFilter) {
                        show = false;
                    }
                    
                    if (dateToFilter && activityDateStr > dateToFilter) {
                        show = false;
                    }
                }
            } catch (error) {
                console.error('Lỗi format ngày:', error, lastActivity);
                show = false;
            }
        }
        
        row.style.display = show ? 'table-row' : 'none';
        
        // Ẩn/hiện chi tiết sessions tương ứng
        const detailRows = document.querySelectorAll(`.session-detail[data-domain="${domain}"][data-user="${user}"]`);
        detailRows.forEach(detailRow => {
            detailRow.style.display = 'none'; // Luôn ẩn chi tiết khi lọc
        });
    });
}

// Function để xóa tất cả bộ lọc
function clearFilters() {
    document.getElementById('domainFilter').value = '';
    document.getElementById('userFilter').value = '';
    document.getElementById('statusFilter').value = '';
    document.getElementById('dateFromFilter').value = '';
    document.getElementById('dateToFilter').value = '';
    
    // Hiển thị tất cả rows
    const rows = document.querySelectorAll('.session-row');
    rows.forEach(row => {
        row.style.display = 'table-row';
    });
}

// Function để mở rộng/xem chi tiết user sessions
function expandUserSessions(domain, user) {
    const sessionRows = document.querySelectorAll(`.session-detail[data-domain="${domain}"][data-user="${user}"]`);
    const userRow = document.querySelector(`.session-row[data-domain="${domain}"][data-user="${user}"]`);
    const expandBtn = userRow.querySelector('.btn-secondary i');
    
    sessionRows.forEach(row => {
        if (row.style.display === 'none') {
            row.style.display = 'table-row';
            expandBtn.className = 'fas fa-compress';
        } else {
            row.style.display = 'none';
            expandBtn.className = 'fas fa-expand';
        }
    });
}

// Function để xem tất cả sessions của một user
function viewUserSessions(domain, user) {
    const sessionRows = document.querySelectorAll(`.session-detail[data-domain="${domain}"][data-user="${user}"]`);
    const sessionIds = [];
    
    sessionRows.forEach(sessionRow => {
        const sessionId = sessionRow.querySelector('td:first-child small').textContent;
        sessionIds.push(sessionId);
    });
    
    // Hiển thị modal với tất cả sessions của user
    showGroupSessionsModal(`User: ${user} (${domain})`, sessionIds);
}

// Function để mở rộng/xem chi tiết user sessions
function expandUserSessions(domain, user) {
    const sessionRows = document.querySelectorAll(`.session-detail[data-domain="${domain}"][data-user="${user}"]`);
    const userRow = document.querySelector(`.session-row[data-domain="${domain}"][data-user="${user}"]`);
    const expandBtn = userRow.querySelector('.btn-secondary i');
    
    sessionRows.forEach(row => {
        if (row.style.display === 'none') {
            row.style.display = 'table-row';
            expandBtn.className = 'fas fa-compress';
        } else {
            row.style.display = 'none';
            expandBtn.className = 'fas fa-expand';
        }
    });
}

// Function để xem tất cả sessions của một user
function viewUserSessions(domain, user) {
    const sessionRows = document.querySelectorAll(`.session-detail[data-domain="${domain}"][data-user="${user}"]`);
    const sessionIds = [];
    
    sessionRows.forEach(sessionRow => {
        const sessionId = sessionRow.querySelector('td:first-child small').textContent;
        sessionIds.push(sessionId);
    });
    
    // Hiển thị modal với tất cả sessions của user
    showGroupSessionsModal(`User: ${user} (${domain})`, sessionIds);
}

// Function để hiển thị modal group sessions
function showGroupSessionsModal(title, sessionIds) {
    const modalContent = `
        <div class="modal-header">
            <h5>${title} (${sessionIds.length} sessions)</h5>
            <button type="button" class="close" onclick="closeGroupModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div class="session-list">
                ${sessionIds.map(id => `
                    <div class="session-item">
                        <span class="session-id">${id}</span>
                        <button class="btn btn-sm btn-primary" onclick="viewChatHistory('${id}'); closeGroupModal();">
                            <i class="fas fa-eye"></i> Xem Chat
                        </button>
                    </div>
                `).join('')}
            </div>
        </div>
    `;
    
    const modal = document.getElementById('groupSessionsModal');
    modal.querySelector('.modal-content').innerHTML = modalContent;
    modal.classList.add('show');
}

function closeGroupModal() {
    document.getElementById('groupSessionsModal').classList.remove('show');
}

// Function để set filter ngày nhanh
function setDateFilter(type) {
    const today = new Date();
    let fromDate, toDate;
    
    switch(type) {
        case 'today':
            // Hôm nay
            fromDate = new Date(today);
            toDate = new Date(today);
            break;
        case 'week':
            // Tuần này (từ thứ 2 đến chủ nhật)
            const dayOfWeek = today.getDay();
            const daysToMonday = dayOfWeek === 0 ? 6 : dayOfWeek - 1;
            fromDate = new Date(today);
            fromDate.setDate(today.getDate() - daysToMonday);
            toDate = new Date(today);
            break;
        case 'month':
            // Tháng này
            fromDate = new Date(today.getFullYear(), today.getMonth(), 1);
            toDate = new Date(today);
            break;
    }
    
    // Format dates to YYYY-MM-DD
    const formatDate = (date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    };
    
    // Set giá trị cho date inputs
    const fromDateStr = formatDate(fromDate);
    const toDateStr = formatDate(toDate);
    
    document.getElementById('dateFromFilter').value = fromDateStr;
    document.getElementById('dateToFilter').value = toDateStr;
    
    // Debug: Kiểm tra format ngày
    console.log('Debug setDateFilter:', {
        type: type,
        fromDate: fromDate,
        toDate: toDate,
        fromDateStr: fromDateStr,
        toDateStr: toDateStr
    });
    
    // Hiển thị thông báo ngày đã chọn
    const formatDateDisplay = (date) => {
        return date.toLocaleDateString('vi-VN', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric'
        });
    };
    
    // Hiển thị thông báo
    const message = `Đã chọn: ${formatDateDisplay(fromDate)} - ${formatDateDisplay(toDate)}`;
    Dashboard.getInstance().showToast(message, 'success', 'Bộ lọc ngày');
    
    // Áp dụng filter
    filterSessions();
    
    // Debug: Hiển thị thông tin về sessions
    const visibleRows = document.querySelectorAll('.session-row[style*="table-row"]');
    const allRows = document.querySelectorAll('.session-row');
    console.log(`Sau khi lọc: ${visibleRows.length}/${allRows.length} sessions hiển thị`);
    
    // Debug: Hiển thị thông tin chi tiết về tất cả sessions
    allRows.forEach((row, index) => {
        const domain = row.dataset.domain;
        const user = row.dataset.user;
        const lastActivity = row.dataset.lastActivity;
        const isVisible = row.style.display !== 'none';
        
        // Debug timezone info
        if (lastActivity) {
            const activityDate = new Date(lastActivity);
            const utcDate = activityDate.toISOString().split('T')[0];
            const localDate = `${activityDate.getFullYear()}-${String(activityDate.getMonth() + 1).padStart(2, '0')}-${String(activityDate.getDate()).padStart(2, '0')}`;
            
            console.log(`Session ${index + 1}:`, {
                domain: domain,
                user: user,
                lastActivity: lastActivity,
                utcDate: utcDate,
                localDate: localDate,
                isVisible: isVisible
            });
        } else {
            console.log(`Session ${index + 1}:`, {
                domain: domain,
                user: user,
                lastActivity: lastActivity,
                isVisible: isVisible
            });
        }
    });

}

// Helper methods cho delete functions
function getUserSessionCount(domain, user) {
    const dashboard = Dashboard.getInstance();
    if (!dashboard.originalSessionsData) return 0;
    
    const userSessions = dashboard.originalSessionsData.filter(session => {
        const userInfo = session.user_info ? (typeof session.user_info === 'string' ? JSON.parse(session.user_info) : session.user_info) : {};
        const sessionDomain = userInfo.domain || 'Unknown';
        const sessionUser = userInfo.user || 'Anonymous';
        return sessionDomain === domain && sessionUser === user;
    });
    
    return userSessions.length;
}

function getUserSessionIds(domain, user) {
    const dashboard = Dashboard.getInstance();
    if (!dashboard.originalSessionsData) return [];
    
    const userSessions = dashboard.originalSessionsData.filter(session => {
        const userInfo = session.user_info ? (typeof session.user_info === 'string' ? JSON.parse(session.user_info) : session.user_info) : {};
        const sessionDomain = userInfo.domain || 'Unknown';
        const sessionUser = userInfo.user || 'Anonymous';
        return sessionDomain === domain && sessionUser === user;
    });
    
    return userSessions.map(session => session.session_id);
}

// Initialize dashboard
let dashboardInstance = null;

Dashboard.getInstance = function() {
    if (!dashboardInstance) {
        dashboardInstance = new Dashboard();
    }
    return dashboardInstance;
};

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    Dashboard.getInstance();
});

// ==================== Q&A Database Helper Functions ====================

// Function để lọc dữ liệu Q&A
function filterQAData() {
    Dashboard.getInstance().filterQAData();
}

// Function để xóa bộ lọc Q&A
function clearQAFilters() {
    document.getElementById('qaSearchInput').value = '';
    document.getElementById('qaCategoryFilter').value = '';
    document.getElementById('qaStatusFilter').value = '';
    Dashboard.getInstance().filterQAData();
    Dashboard.getInstance().showToast('Đã xóa tất cả bộ lọc', 'info', 'Q&A Database');
}

// ==================== Chat Sessions Delete Functions ====================

// Function để xóa một session
async function deleteSession(sessionId) {
    if (!confirm('Bạn có chắc chắn muốn xóa session này? Hành động này không thể hoàn tác.')) {
        return;
    }

    try {
        const response = await fetch(`/api/statistics/chat-sessions/${sessionId}`, {
            method: 'DELETE'
        });

        if (response.ok) {
            const result = await response.json();
            if (result.success) {
                Dashboard.getInstance().showToast('Đã xóa session thành công', 'success', 'Xóa Session');
                // Reload chat sessions
                Dashboard.getInstance().loadChatSessions();
            } else {
                Dashboard.getInstance().showToast(result.error || 'Lỗi xóa session', 'error', 'Xóa Session');
            }
        } else {
            Dashboard.getInstance().showToast('Lỗi kết nối server', 'error', 'Xóa Session');
        }
    } catch (error) {
        console.error('Lỗi xóa session:', error);
        Dashboard.getInstance().showToast('Lỗi xóa session', 'error', 'Xóa Session');
    }
}

// Function để xóa tất cả sessions của một user
async function deleteUserSessions(domain, user) {
    const sessionCount = getUserSessionCount(domain, user);
    
    if (!confirm(`Bạn có chắc chắn muốn xóa tất cả ${sessionCount} sessions của user "${user}" từ domain "${domain}"? Hành động này không thể hoàn tác.`)) {
        return;
    }

    try {
        // Lấy danh sách session IDs của user
        const sessionIds = getUserSessionIds(domain, user);
        
        const response = await fetch('/api/statistics/chat-sessions', {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ sessionIds })
        });

        if (response.ok) {
            const result = await response.json();
            if (result.success) {
                Dashboard.getInstance().showToast(`Đã xóa ${result.data.deleted} sessions thành công${result.data.failed > 0 ? `, ${result.data.failed} thất bại` : ''}`, 'success', 'Xóa Sessions');
                // Reload chat sessions
                Dashboard.getInstance().loadChatSessions();
            } else {
                Dashboard.getInstance().showToast(result.error || 'Lỗi xóa sessions', 'error', 'Xóa Sessions');
            }
        } else {
            Dashboard.getInstance().showToast('Lỗi kết nối server', 'error', 'Xóa Sessions');
        }
    } catch (error) {
        console.error('Lỗi xóa user sessions:', error);
        Dashboard.getInstance().showToast('Lỗi xóa sessions', 'error', 'Xóa Sessions');
    }
}
