const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

// Model cho API Mapping
const ApiMapping = sequelize.define('ApiMapping', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  domain: {
    type: DataTypes.STRING(255),
    allowNull: false,
    unique: true
  },
  api_server: {
    type: DataTypes.STRING(500),
    allowNull: false
  },
  environment: {
    type: DataTypes.STRING(50),
    allowNull: false,
    defaultValue: 'development'
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  is_active: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: true
  }
}, {
  tableName: 'api_mappings',
  timestamps: true
});

// Model cho Q&A Database
const QADatabase = sequelize.define('QADatabase', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  question: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  answer: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  category: {
    type: DataTypes.STRING(100),
    allowNull: true
  },
  keywords: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  score: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  is_active: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: true
  }
}, {
  tableName: 'qa_database',
  timestamps: true
});

// Model cho Chat Sessions
const ChatSession = sequelize.define('ChatSession', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  session_id: {
    type: DataTypes.STRING(255),
    allowNull: false,
    unique: true
  },
  thread_id: {
    type: DataTypes.STRING(255),
    allowNull: true
  },
  user_info: {
    type: DataTypes.JSON,
    allowNull: true
  },
  api_mapping: {
    type: DataTypes.JSON,
    allowNull: true
  },
  last_activity: {
    type: DataTypes.DATE,
    allowNull: false,
    defaultValue: DataTypes.NOW
  },
  is_active: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: true
  }
}, {
  tableName: 'chat_sessions',
  timestamps: true
});

// Model cho Chat Messages
const ChatMessage = sequelize.define('ChatMessage', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  session_id: {
    type: DataTypes.STRING(255),
    allowNull: false
  },
  message_type: {
    type: DataTypes.ENUM('user', 'bot', 'api_call', 'qa_response', 'error'),
    allowNull: false
  },
  content: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  response: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  api_type: {
    type: DataTypes.STRING(50),
    allowNull: true
  },
  api_params: {
    type: DataTypes.JSON,
    allowNull: true
  },
  api_result: {
    type: DataTypes.JSON,
    allowNull: true
  },
  processing_time: {
    type: DataTypes.INTEGER, // milliseconds
    allowNull: true
  },
  domain: {
    type: DataTypes.STRING(255),
    allowNull: true
  },
  user_agent: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  ip_address: {
    type: DataTypes.STRING(45),
    allowNull: true
  },
  error: {
    type: DataTypes.TEXT,
    allowNull: true
  }
}, {
  tableName: 'chat_messages',
  timestamps: true,
  indexes: [
    {
      fields: ['session_id']
    },
    {
      fields: ['created_at']
    },
    {
      fields: ['message_type']
    }
  ]
});

// Model cho Chat Statistics
const ChatStatistic = sequelize.define('ChatStatistic', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  date: {
    type: DataTypes.DATEONLY,
    allowNull: false,
    unique: true
  },
  total_sessions: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  total_messages: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  api_calls: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  qa_responses: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  gpt_responses: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  avg_response_time: {
    type: DataTypes.FLOAT,
    allowNull: false,
    defaultValue: 0
  },
  unique_users: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  }
}, {
  tableName: 'chat_statistics',
  timestamps: true
});

// Relationships
ChatSession.hasMany(ChatMessage, { foreignKey: 'session_id', sourceKey: 'session_id' });
ChatMessage.belongsTo(ChatSession, { foreignKey: 'session_id', targetKey: 'session_id' });

module.exports = {
  ApiMapping,
  QADatabase,
  ChatSession,
  ChatMessage,
  ChatStatistic
};
