// QA Manager JavaScript - Hỗ trợ lưu dữ liệu trực tiếp
class QAManager {
    constructor() {
        this.qaData = [];
        this.filePath = 'kthc-qa-database.json';
        this.backupPath = 'kthc-qa-database-backup.json';
    }

    // Tải dữ liệu từ API
    async loadData() {
        try {
            const response = await fetch('/api/qa-data');
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            const data = await response.json();
            this.qaData = data.kthc_qa_database || [];
            return this.qaData;
        } catch (error) {
            console.error('Lỗi khi tải dữ liệu:', error);
            throw error;
        }
    }

    // Lưu dữ liệu vào server
    async saveData() {
        try {
            const data = {
                kthc_qa_database: this.qaData,
                metadata: {
                    last_updated: new Date().toISOString(),
                    total_questions: this.qaData.length,
                    categories: [...new Set(this.qaData.map(qa => qa.category))],
                    total_keywords: [...new Set(this.qaData.flatMap(qa => qa.keywords || []))].length
                }
            };

            const response = await fetch('/api/save-qa-data', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error('Lỗi khi lưu dữ liệu:', error);
            throw error;
        }
    }

    // Tạo backup
    async createBackup() {
        try {
            const backupData = {
                kthc_qa_database: [...this.qaData],
                backup_info: {
                    created_at: new Date().toISOString(),
                    original_file: this.filePath,
                    total_items: this.qaData.length
                }
            };

            const response = await fetch('/api/create-backup', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(backupData)
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error('Lỗi khi tạo backup:', error);
            throw error;
        }
    }

    // Thêm câu hỏi mới
    addQuestion(question, answer, category, keywords = []) {
        const newQA = {
            question: question.trim(),
            answer: answer.trim(),
            category: category,
            keywords: keywords.filter(k => k.trim() !== ''),
            created_at: new Date().toISOString(),
            id: this.generateId()
        };

        this.qaData.push(newQA);
        return newQA;
    }

    // Cập nhật câu hỏi
    updateQuestion(id, question, answer, category, keywords = []) {
        const index = this.qaData.findIndex(qa => qa.id === id);
        if (index === -1) {
            throw new Error('Không tìm thấy câu hỏi');
        }

        this.qaData[index] = {
            ...this.qaData[index],
            question: question.trim(),
            answer: answer.trim(),
            category: category,
            keywords: keywords.filter(k => k.trim() !== ''),
            updated_at: new Date().toISOString()
        };

        return this.qaData[index];
    }

    // Xóa câu hỏi
    deleteQuestion(id) {
        const index = this.qaData.findIndex(qa => qa.id === id);
        if (index === -1) {
            throw new Error('Không tìm thấy câu hỏi');
        }

        const deletedQA = this.qaData.splice(index, 1)[0];
        return deletedQA;
    }

    // Tìm kiếm câu hỏi
    searchQuestions(query, category = null) {
        const searchTerm = query.toLowerCase();
        
        return this.qaData.filter(qa => {
            const matchesSearch = qa.question.toLowerCase().includes(searchTerm) || 
                                qa.answer.toLowerCase().includes(searchTerm) ||
                                (qa.keywords || []).some(keyword => 
                                    keyword.toLowerCase().includes(searchTerm)
                                );
            
            const matchesCategory = !category || qa.category === category;
            
            return matchesSearch && matchesCategory;
        });
    }

    // Lấy thống kê
    getStats() {
        const categories = new Set(this.qaData.map(qa => qa.category));
        const allKeywords = this.qaData.flatMap(qa => qa.keywords || []);
        const uniqueKeywords = new Set(allKeywords);

        return {
            totalQuestions: this.qaData.length,
            totalCategories: categories.size,
            totalKeywords: uniqueKeywords.size,
            categories: Array.from(categories),
            topKeywords: this.getTopKeywords(),
            recentQuestions: this.getRecentQuestions(5)
        };
    }

    // Lấy từ khóa phổ biến
    getTopKeywords(limit = 10) {
        const keywordCount = {};
        
        this.qaData.forEach(qa => {
            (qa.keywords || []).forEach(keyword => {
                keywordCount[keyword] = (keywordCount[keyword] || 0) + 1;
            });
        });

        return Object.entries(keywordCount)
            .sort(([,a], [,b]) => b - a)
            .slice(0, limit)
            .map(([keyword, count]) => ({ keyword, count }));
    }

    // Lấy câu hỏi gần đây
    getRecentQuestions(limit = 5) {
        return this.qaData
            .sort((a, b) => new Date(b.created_at || 0) - new Date(a.created_at || 0))
            .slice(0, limit);
    }

    // Validate dữ liệu
    validateQA(qa) {
        const errors = [];

        if (!qa.question || qa.question.trim().length < 5) {
            errors.push('Câu hỏi phải có ít nhất 5 ký tự');
        }

        if (!qa.answer || qa.answer.trim().length < 10) {
            errors.push('Câu trả lời phải có ít nhất 10 ký tự');
        }

        if (!qa.category) {
            errors.push('Phải chọn danh mục');
        }

        if (qa.keywords && qa.keywords.length > 10) {
            errors.push('Tối đa 10 từ khóa');
        }

        return {
            isValid: errors.length === 0,
            errors: errors
        };
    }

    // Tạo ID duy nhất
    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    }

    // Export dữ liệu
    exportData(format = 'json') {
        const data = {
            kthc_qa_database: this.qaData,
            export_info: {
                exported_at: new Date().toISOString(),
                format: format,
                total_items: this.qaData.length
            }
        };

        switch (format.toLowerCase()) {
            case 'json':
                return JSON.stringify(data, null, 2);
            case 'csv':
                return this.convertToCSV(data.kthc_qa_database);
            default:
                throw new Error('Format không được hỗ trợ');
        }
    }

    // Chuyển đổi sang CSV
    convertToCSV(data) {
        const headers = ['question', 'answer', 'category', 'keywords', 'created_at'];
        const csvContent = [
            headers.join(','),
            ...data.map(qa => [
                `"${qa.question.replace(/"/g, '""')}"`,
                `"${qa.answer.replace(/"/g, '""')}"`,
                `"${qa.category}"`,
                `"${(qa.keywords || []).join('; ')}"`,
                `"${qa.created_at || ''}"`
            ].join(','))
        ].join('\n');

        return csvContent;
    }

    // Import dữ liệu
    importData(data, merge = false) {
        try {
            let importedData;
            
            if (typeof data === 'string') {
                importedData = JSON.parse(data);
            } else {
                importedData = data;
            }

            const qaArray = importedData.kthc_qa_database || importedData;
            
            if (!Array.isArray(qaArray)) {
                throw new Error('Dữ liệu không đúng định dạng');
            }

            if (merge) {
                // Merge với dữ liệu hiện tại
                this.qaData = [...this.qaData, ...qaArray];
            } else {
                // Thay thế dữ liệu hiện tại
                this.qaData = qaArray;
            }

            // Thêm ID cho các câu hỏi chưa có ID
            this.qaData.forEach(qa => {
                if (!qa.id) {
                    qa.id = this.generateId();
                }
            });

            return {
                success: true,
                imported: qaArray.length,
                total: this.qaData.length
            };
        } catch (error) {
            console.error('Lỗi khi import dữ liệu:', error);
            throw error;
        }
    }

    // Tìm câu hỏi tương tự
    findSimilarQuestions(question, limit = 5) {
        const questionWords = question.toLowerCase().split(/\s+/);
        
        const similarities = this.qaData.map(qa => {
            const qaWords = qa.question.toLowerCase().split(/\s+/);
            const commonWords = questionWords.filter(word => 
                qaWords.some(qaWord => qaWord.includes(word) || word.includes(qaWord))
            );
            
            return {
                qa: qa,
                similarity: commonWords.length / Math.max(questionWords.length, qaWords.length)
            };
        });

        return similarities
            .filter(item => item.similarity > 0.1)
            .sort((a, b) => b.similarity - a.similarity)
            .slice(0, limit)
            .map(item => ({
                ...item.qa,
                similarity: Math.round(item.similarity * 100)
            }));
    }
}

// Export cho sử dụng trong browser
if (typeof window !== 'undefined') {
    window.QAManager = QAManager;
}

// Export cho Node.js
if (typeof module !== 'undefined' && module.exports) {
    module.exports = QAManager;
}
