// QA Search Engine - Tìm kiếm câu trả lời trong database
class QASearchEngine {
    constructor() {
        this.qaDatabase = null;
        this.loadDatabase();
    }

    // Load database từ file JSON
    async loadDatabase() {
        try {
            const response = await fetch('./qa-database.json');
            this.qaDatabase = await response.json();
            console.log('QA Database loaded successfully');
        } catch (error) {
            console.error('Error loading QA database:', error);
            this.qaDatabase = { qa_pairs: [] };
        }
    }

    // Tự động extract keywords từ câu hỏi
    extractKeywords(question) {
        const stopWords = ['tôi', 'muốn', 'làm', 'xem', 'để', 'có', 'thể', 'giúp', 'cho', 'với', 'và', 'hoặc', 'của', 'trong', 'từ', 'đến'];
        
        // Chuyển về lowercase và tách từ
        let words = question.toLowerCase()
            .replace(/[.,?!]/g, '')
            .split(/\s+/)
            .filter(word => word.length > 1);
        
        // Loại bỏ stop words
        words = words.filter(word => !stopWords.includes(word));
        
        // Thêm các từ ghép phổ biến
        const compoundWords = this.extractCompoundWords(question);
        words = [...words, ...compoundWords];
        
        return [...new Set(words)]; // Loại bỏ duplicate
    }

    // Extract từ ghép phổ biến
    extractCompoundWords(question) {
        const compoundPatterns = [
            'tài sản cố định',
            'báo cáo',
            'chấm công',
            'phòng ban',
            'hệ thống',
            'thống kê',
            'tổng hợp',
            'xuất báo cáo',
            'tính lương',
            'tiền lương',
            'lương tháng',
            'phân quyền',
            'người dùng',
            'quyền hạn'
        ];
        
        const found = [];
        compoundPatterns.forEach(pattern => {
            if (question.toLowerCase().includes(pattern)) {
                found.push(pattern);
            }
        });
        
        return found;
    }

    // Tìm kiếm câu trả lời dựa trên câu hỏi
    searchAnswer(userQuestion) {
        if (!this.qaDatabase || !this.qaDatabase.qa_pairs) {
            return null;
        }

        const userKeywords = this.extractKeywords(userQuestion);
        let bestMatch = null;
        let bestScore = 0;

        // Tìm kiếm trong database
        this.qaDatabase.qa_pairs.forEach(qa => {
            const score = this.calculateMatchScore(userKeywords, qa.keywords, userQuestion, qa.question);
            
            if (score > bestScore) {
                bestScore = score;
                bestMatch = qa;
            }
        });

        // Trả về kết quả nếu score đủ cao (threshold = 0.3)
        if (bestScore >= 0.3) {
            // Xử lý answer với path nếu có
            let processedAnswer = bestMatch.answer;
            if (bestMatch.path) {
                const baseUrl = window.location.origin || 'https://muvietplus.net';
                const fullUrl = bestMatch.path.startsWith('http') ? bestMatch.path : `${baseUrl}${bestMatch.path}`;
                processedAnswer += `\n- Truy cập: <a href='${fullUrl}' target='_blank'>Truy cập tại đây</a>`;
            }

            return {
                answer: processedAnswer,
                score: bestScore,
                matched_question: bestMatch.question,
                matched_keywords: bestMatch.keywords,
                path: bestMatch.path
            };
        }

        return null;
    }

    // Tính điểm match giữa câu hỏi user và câu hỏi trong database
    calculateMatchScore(userKeywords, dbKeywords, userQuestion, dbQuestion) {
        let score = 0;
        
        // 1. Keyword matching (60% trọng số)
        const keywordMatches = userKeywords.filter(keyword => 
            dbKeywords.some(dbKeyword => 
                dbKeyword.toLowerCase().includes(keyword.toLowerCase()) ||
                keyword.toLowerCase().includes(dbKeyword.toLowerCase())
            )
        );
        
        const keywordScore = keywordMatches.length / Math.max(userKeywords.length, dbKeywords.length);
        score += keywordScore * 0.6;
        
        // 2. Exact question matching (30% trọng số)
        const userLower = userQuestion.toLowerCase();
        const dbLower = dbQuestion.toLowerCase();
        
        if (userLower === dbLower) {
            score += 0.3;
        } else if (userLower.includes(dbLower) || dbLower.includes(userLower)) {
            score += 0.2;
        }
        
        // 3. Partial word matching (10% trọng số)
        const userWords = userQuestion.toLowerCase().split(/\s+/);
        const dbWords = dbQuestion.toLowerCase().split(/\s+/);
        
        const commonWords = userWords.filter(word => 
            dbWords.some(dbWord => 
                dbWord.includes(word) || word.includes(dbWord)
            )
        );
        
        const partialScore = commonWords.length / Math.max(userWords.length, dbWords.length);
        score += partialScore * 0.1;
        
        return score;
    }

    // Thêm câu hỏi mới vào database
    addNewQA(question, answer, customKeywords = null, path = null) {
        if (!this.qaDatabase) {
            this.qaDatabase = { qa_pairs: [], metadata: {} };
        }

        // Tự động chuyển đổi đường dẫn thành link
        const processedAnswer = this.convertPathsToLinks(answer, path);

        const keywords = customKeywords || this.extractKeywords(question);
        const newQA = {
            id: this.qaDatabase.qa_pairs.length + 1,
            question: question,
            keywords: keywords,
            answer: processedAnswer,
            path: path || null
        };

        this.qaDatabase.qa_pairs.push(newQA);
        this.updateMetadata();
        
        return newQA;
    }

    // Chuyển đổi đường dẫn thành link
    convertPathsToLinks(text, path = null) {
        let processedText = text;
        
        // Nếu có path riêng biệt, thêm vào cuối
        if (path && path.trim()) {
            const baseUrl = window.location.origin || 'https://muvietplus.net';
            const fullUrl = path.startsWith('http') ? path : `${baseUrl}${path}`;
            processedText += `\n- Truy cập: <a href='${fullUrl}' target='_blank'>Truy cập tại đây</a>`;
        } else {
            // Tìm và thay thế các pattern "Truy cập: /path" thành link
            const baseUrl = window.location.origin || 'https://muvietplus.net';
            processedText = text.replace(
                /Truy cập:\s*(\/[^\s\n]+)/g, 
                `Truy cập: <a href='${baseUrl}$1' target='_blank'>Truy cập tại đây</a>`
            );
        }
        
        return processedText;
    }

    // Cập nhật metadata
    updateMetadata() {
        if (this.qaDatabase) {
            this.qaDatabase.metadata = {
                total_questions: this.qaDatabase.qa_pairs.length,
                last_updated: new Date().toISOString().split('T')[0],
                version: "1.0",
                description: "Database câu hỏi và câu trả lời cho AI Assistant"
            };
        }
    }

    // Export database ra file
    exportDatabase() {
        if (this.qaDatabase) {
            const dataStr = JSON.stringify(this.qaDatabase, null, 2);
            const dataBlob = new Blob([dataStr], {type: 'application/json'});
            const url = URL.createObjectURL(dataBlob);
            const link = document.createElement('a');
            link.href = url;
            link.download = 'qa-database-updated.json';
            link.click();
        }
    }

    // Debug: In ra thông tin tìm kiếm
    debugSearch(userQuestion) {
        const userKeywords = this.extractKeywords(userQuestion);
        console.log('User Question:', userQuestion);
        console.log('Extracted Keywords:', userKeywords);
        
        const result = this.searchAnswer(userQuestion);
        if (result) {
            console.log('Found Answer:', result);
        } else {
            console.log('No match found');
        }
        
        return result;
    }
}

// Sử dụng trong chatbot
function integrateWithChatbot() {
    const qaEngine = new QASearchEngine();
    
    // Hàm xử lý tin nhắn từ user
    async function handleUserMessage(userMessage) {
        // Đợi database load xong
        await new Promise(resolve => setTimeout(resolve, 100));
        
        // Tìm kiếm trong QA database trước
        const qaResult = qaEngine.searchAnswer(userMessage);
        
        if (qaResult) {
            // Trả về câu trả lời từ database
            return {
                source: 'qa_database',
                answer: qaResult.answer,
                confidence: qaResult.score,
                matched_question: qaResult.matched_question
            };
        } else {
            // Không tìm thấy, gọi AI
            return {
                source: 'ai_assistant',
                answer: null, // Sẽ được fill bởi AI
                confidence: 0
            };
        }
    }
    
    return {
        qaEngine,
        handleUserMessage
    };
}

// Export cho sử dụng
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { QASearchEngine, integrateWithChatbot };
}
