require('dotenv').config();
const express = require('express');
const cors = require('cors');

// Import database configurations
const { 
  sequelize, 
  testConnection, 
  syncDatabase 
} = require('./config/database');

// Import database services
const {
  ApiMappingService,
  QADatabaseService,
  ChatSessionService,
  StatisticsService
} = require('./services/databaseService');
const { ensureConnection } = require('./config/database');

// Import API mapping service
const { 
  getApiServerByDomain, 
  getEnvironmentInfo,
  getAllDomains 
} = require('./utils/apiMappingService');

// Import KTHC config
const { setKthcBaseUrl } = require('./config/kthc');

// Import middleware
const { sessionMiddleware, kiemtraSession } = require('./middleware/sessionMiddleware');
const { kiemtraApiMiddleware, kiemtraApiCallMiddleware } = require('./middleware/apiMiddleware');

// Import services
const { xulyBotGPT, layLichsuchat } = require('./utils/chatGPTService');
const { taoLuongvoiCTGS, taoLuongvoiCTGSDachon } = require('./api/taoluong/luongService');
const { taoBaohiem, taoBaohiemvoiCTGSDachon } = require('./api/taobaohiem/baoHiemService');
const { taoCongdoan, taoCongdoanvoiCTGSDachon } = require('./api/taocongdoan/congDoanService');
const { taoCTGS } = require('./api/ctgs/ctgsService');

// Import routes
const luongRoutes = require('./api/taoluong/luongRoutes');
const baoHiemRoutes = require('./api/taobaohiem/baoHiemRoutes');
const congDoanRoutes = require('./api/taocongdoan/congDoanRoutes');
const ctgsRoutes = require('./api/ctgs/ctgsRoutes');
const statisticsRoutes = require('./api/statistics');

const app = express();
const PORT = process.env.PORT || 3001;

// CORS Configuration
const corsOptions = {
  origin: function (origin, callback) {
    // Allow requests with no origin (like mobile apps or curl requests)
    if (!origin) {
      console.log('✅ CORS allowed (no origin)');
      return callback(null, true);
    }
    
    // Allow all origins for development
    const allowedOrigins = [
      'http://localhost:3000',
      'http://localhost:3001',
      'http://127.0.0.1:3000',
      'http://127.0.0.1:3001',
      'https://api.muvietplus.net',
      'https://kthcdemo.muvietplus.net',
      'https://kthcdemo.dtsoft.vn',
      'https://kthctravinh.dtsoft.vn',
      'https://kthccantho.dtsoft.vn',
      'https://kthchcm.dtsoft.vn',
      'https://kthcdanang.dtsoft.vn',
      'https://muvietplus.net',
      'https://www.muvietplus.net',
      // Add your static site domain here
      'https://your-static-site.com',
      'https://*.github.io',
      'https://*.netlify.app',
      'https://*.vercel.app'
    ];
    
    // Check if origin is allowed
    const isAllowed = allowedOrigins.some(allowedOrigin => {
      if (allowedOrigin.includes('*')) {
        return origin.includes(allowedOrigin.replace('*', ''));
      }
      // Allow exact match or subdomain/path match
      return origin === allowedOrigin || origin.startsWith(allowedOrigin + '/');
    });
    
    if (isAllowed) {
      console.log('✅ CORS allowed origin:', origin);
      callback(null, true);
    } else {
      console.log('🚫 CORS blocked origin:', origin);
      console.log('📋 Allowed origins:', allowedOrigins);
      // Allow anyway for development
      console.log('🔄 Allowing blocked origin for development');
      callback(null, true);
    }
  },
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With', 'Accept', 'Origin'],
  optionsSuccessStatus: 200
};

// Middleware
app.use(cors(corsOptions));
app.use(express.json());

// Handle OPTIONS requests (preflight)
app.options('*', cors(corsOptions));

// Serve static files without API mapping
app.use(express.static('public'));

// Serve kthcdemo files
app.use('/kthcdemo', express.static('kthcdemo'));

// Dynamic API Mapping Middleware - chỉ áp dụng cho API routes
app.use('/api', async (req, res, next) => {
  try {
    // Đảm bảo kết nối database
    const isConnected = await ensureConnection();
    if (!isConnected) {
      console.log('⚠️ Không thể kết nối database, sử dụng fallback');
      req.apiMapping = {
        domain: 'localhost',
        apiServer: process.env.KTHC_BASE_URL || 'https://kthcdemoapi.dtsoft.vn/api',
        environment: 'development',
        description: 'Database fallback'
      };
      setKthcBaseUrl(req.apiMapping.apiServer);
      return next();
    }

    const origin = req.headers.origin || req.headers.host;
    let domain;
    
    // Parse domain from origin or host
    if (origin) {
      try {
        domain = new URL(origin).hostname;
      } catch (urlError) {
        // If origin is not a valid URL, use it as domain
        domain = origin;
      }
    } else {
      domain = req.headers.host;
    }
    
    // Validate domain
    if (!domain || domain.trim() === '') {
      console.warn('⚠️ Không thể xác định domain, sử dụng localhost');
      domain = 'localhost';
    }
    
    console.log('🔍 Processing domain:', domain);
    
    // Lấy API mapping từ database
    const apiMapping = await ApiMappingService.getByDomain(domain);
    
    if (apiMapping) {
      console.log('✅ Tìm thấy API mapping:', {
        domain: apiMapping.domain,
        api_server: apiMapping.api_server,
        environment: apiMapping.environment,
        description: apiMapping.description
      });
      
      // Set KTHC baseUrl
      setKthcBaseUrl(apiMapping.api_server);
      
      // Add API mapping info to request
      req.apiMapping = {
        domain: apiMapping.domain,
        apiServer: apiMapping.api_server,
        environment: apiMapping.environment,
        description: apiMapping.description
      };
    } else {
      console.log('❌ Không tìm thấy API mapping cho domain:', domain);
      
      // Fallback to default
      const fallbackUrl = process.env.KTHC_BASE_URL || 'https://kthcdemoapi.dtsoft.vn/api';
      setKthcBaseUrl(fallbackUrl);
      
      req.apiMapping = {
        domain: domain,
        apiServer: fallbackUrl,
        environment: 'development',
        description: 'Default fallback'
      };
    }
    
    console.log('🗺️ API Mapping:', {
      domain: req.apiMapping.domain,
      apiServer: req.apiMapping.apiServer,
      environment: req.apiMapping.environment,
      description: req.apiMapping.description
    });
    
    next();
  } catch (error) {
    console.error('❌ API Mapping Error:', error.message);
    
    // Fallback to localhost for development
    console.log('🔄 Fallback to localhost mapping');
    req.apiMapping = {
      domain: 'localhost',
      apiServer: process.env.KTHC_BASE_URL || 'https://kthcdemoapi.dtsoft.vn/api',
      environment: 'development',
      description: 'Error fallback'
    };
    setKthcBaseUrl(req.apiMapping.apiServer);
    
    next();
  }
});

// Initialize database on startup
const { initializeDatabase } = require('./init-database');
initializeDatabase().then((dbInitialized) => {
  if (dbInitialized) {
    console.log('✅ Sử dụng database storage');
  } else {
    console.log('⚠️ Sử dụng file storage');
  }
});

// Update statistics every hour
setInterval(async () => {
  try {
    await StatisticsService.updateDailyStats();
  } catch (error) {
    console.error('❌ Lỗi cập nhật thống kê:', error);
  }
}, 60 * 60 * 1000);

// API Routes
app.use('/api/luong', luongRoutes);
app.use('/api/bao-hiem', baoHiemRoutes);
app.use('/api/cong-doan', congDoanRoutes);
app.use('/api/ctgs', ctgsRoutes);
app.use('/api/statistics', statisticsRoutes);

// API Mapping Info Endpoint
app.get('/api/mapping-info', (req, res) => {
  try {
    res.json({
      success: true,
      data: {
        current_domain: req.apiMapping.domain,
        api_server: req.apiMapping.apiServer,
        environment: req.apiMapping.environment,
        description: req.apiMapping.description,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: 'Lỗi khi lấy mapping info',
      message: error.message
    });
  }
});

// Get All Available Domains
app.get('/api/domains', (req, res) => {
  try {
    const domains = getAllDomains();
    res.json({
      success: true,
      data: domains
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Test endpoint for CORS (GET)
app.get('/api/test', (req, res) => {
  console.log('🔍 CORS Test Request (GET):');
  console.log('  Origin:', req.headers.origin);
  console.log('  Referer:', req.headers.referer);
  console.log('  User-Agent:', req.headers['user-agent']);
  console.log('  API Mapping:', req.apiMapping);
  
  res.json({
    success: true,
    message: 'CORS test successful! (GET)',
    timestamp: new Date().toISOString(),
    origin: req.headers.origin || 'No origin',
    referer: req.headers.referer || 'No referer',
    method: 'GET',
    apiMapping: req.apiMapping,
    corsHeaders: {
      'Access-Control-Allow-Origin': res.getHeader('Access-Control-Allow-Origin'),
      'Access-Control-Allow-Methods': res.getHeader('Access-Control-Allow-Methods'),
      'Access-Control-Allow-Headers': res.getHeader('Access-Control-Allow-Headers')
    }
  });
});

// Main chat endpoint
app.post('/api/chat', 
  sessionMiddleware,
  kiemtraApiMiddleware,
  kiemtraApiCallMiddleware,
  async (req, res) => {
    try {
      const { sessionId, message, isApiCall = false, apiParams = {} } = req.body;
      
      console.log('📨 Chat request:', { sessionId, message, isApiCall });
      console.log('🗺️ API Mapping:', req.apiMapping);
      
      // Handle test message
      if (message === 'test') {
        return res.json({
          success: true,
          message: 'Kết nối thành công!',
          sessionId: req.sessionId,
          apiMapping: req.apiMapping
        });
      }
      
      // Handle API calls
      if (isApiCall) {
        const userInfo = req.userInfo;
        
        // Route to appropriate service based on message
        switch (message) {
          case 'Tạo lương tháng':
            const salaryResult = await taoLuongvoiCTGS(userInfo, req.sessionId);
            luuApiCallVaoSession(req.sessionId, message, salaryResult, 'luong');
            return res.json(salaryResult);
            
          case 'Tạo bảo hiểm tháng':
            const insuranceResult = await taoBaohiem(userInfo, req.sessionId);
            luuApiCallVaoSession(req.sessionId, message, insuranceResult, 'baohiem');
            return res.json(insuranceResult);
            
          case 'Tạo kinh phí công đoàn tháng':
            const unionResult = await taoCongdoan(userInfo, req.sessionId);
            luuApiCallVaoSession(req.sessionId, message, unionResult, 'congdoan');
            return res.json(unionResult);
            
          default:
            // Check if message is a CTGS index (for salary or insurance creation)
            const ctgsIndex = parseInt(message);
            if (!isNaN(ctgsIndex)) {
              // Lấy context từ apiParams để phân biệt user đang chọn CTGS cho lương hay bảo hiểm
              const currentContext = apiParams.currentContext || 'Tạo lương tháng';
              
              if (currentContext.includes('bảo hiểm')) {
                // Xử lý cho bảo hiểm
                const insuranceCTGSResult = await taoBaohiemvoiCTGSDachon(userInfo, ctgsIndex, req.sessionId);
                luuApiCallVaoSession(req.sessionId, `Chọn CTGS ${ctgsIndex} cho bảo hiểm`, insuranceCTGSResult, 'baohiem_ctgs');
                return res.json(insuranceCTGSResult);
              } else if (currentContext.includes('công đoàn')) {
                // Xử lý cho công đoàn
                const unionCTGSResult = await taoCongdoanvoiCTGSDachon(userInfo, ctgsIndex, req.sessionId);
                luuApiCallVaoSession(req.sessionId, `Chọn CTGS ${ctgsIndex} cho công đoàn`, unionCTGSResult, 'congdoan_ctgs');
                return res.json(unionCTGSResult);
              } else {
                // Xử lý cho lương (mặc định)
                const salaryCTGSResult = await taoLuongvoiCTGSDachon(userInfo, ctgsIndex, req.sessionId);
                luuApiCallVaoSession(req.sessionId, `Chọn CTGS ${ctgsIndex} cho lương`, salaryCTGSResult, 'luong_ctgs');
                return res.json(salaryCTGSResult);
              }
            }
            
            // If not recognized, treat as chat
            const chatResult = await xulyBotGPT(req.sessionId, message);
            return res.json(chatResult);
        }
      } else {
        // Handle regular chat - Check for API keywords first, then Q&A
        const apiKeywords = await checkApiKeywords(message, req.userInfo, req.sessionId);
        
        if (apiKeywords.shouldCallApi) {
          return res.json(apiKeywords.result);
        }
        
        if (apiKeywords.shouldUseQA) {
          // Thêm delay nhỏ để hiển thị typing indicator
          await new Promise(resolve => setTimeout(resolve, 1000));
          return res.json(apiKeywords.result);
        }
        
        // If no API keywords and Q&A found, proceed with ChatGPT
        const chatResult = await xulyBotGPT(req.sessionId, message);
        return res.json(chatResult);
      }
      
    } catch (error) {
      console.error('❌ Error in /api/chat:', error);
      res.status(500).json({
        success: false,
        error: error.message || 'Internal server error'
      });
    }
  }
);

// Get session messages
app.get('/api/session/:sessionId/messages', async (req, res) => {
  try {
    const { sessionId } = req.params;
    const result = await layLichsuchat(sessionId);
    res.json(result);
  } catch (error) {
    console.error('❌ Error getting session messages:', error);
    res.status(500).json({
      success: false,
      error: error.message || 'Internal server error'
    });
  }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({
    success: true,
    message: 'Server is running',
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    apiMapping: req.apiMapping
  });
});

// CORS preflight handler
app.options('*', cors(corsOptions));

// Test endpoint để kiểm tra kết nối (POST)
app.post('/api/test', (req, res) => {
  console.log('🔍 CORS Test Request (POST):');
  console.log('  Origin:', req.headers.origin);
  console.log('  Body:', req.body);
  console.log('  API Mapping:', req.apiMapping);
  
  res.json({
    success: true,
    message: 'API test thành công! (POST)',
    timestamp: new Date().toISOString(),
    method: 'POST',
    body: req.body,
    origin: req.headers.origin || 'No origin',
    apiMapping: req.apiMapping
  });
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('❌ Server Error:', err);
  
  // Handle CORS errors specifically
  if (err.message === 'Not allowed by CORS') {
    return res.status(403).json({
      success: false,
      error: 'CORS Error: Origin not allowed',
      details: {
        origin: req.headers.origin,
        allowedOrigins: 'Check server logs for allowed origins'
      }
    });
  }
  
  res.status(500).json({
    success: false,
    error: err.message || 'Internal server error',
    timestamp: new Date().toISOString()
  });
});

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({
    success: false,
    error: 'Endpoint không tồn tại',
    path: req.originalUrl,
    method: req.method
  });
});

// ===== PHẦN 1: ĐỊNH NGHĨA TỪ KHÓA THEO API =====
const API_KEYWORDS = {
  // API Tạo Lương
  luong: {
    hanhdong: ['tạo', 'tao', 'thêm', 'them', 'tính', 'tinh', 'hạch toán', 'hach toan', 'xử lý', 'xu ly', 'sinh', 'generate'],
    doituong: ['lương', 'luong', 'lương tháng', 'luong thang', 'salary', 'payroll', 'lương tháng', 'luong thang'],
    mota: 'Tạo lương tháng'
  },
  
  // API Tạo Bảo Hiểm
  baohiem: {
    hanhdong: ['tạo', 'tao', 'thêm', 'them', 'tính', 'tinh', 'hạch toán', 'hach toan', 'xử lý', 'xu ly', 'sinh', 'generate'],
    doituong: ['bảo hiểm', 'bao hiem', 'bảo hiểm tháng', 'bao hiem thang', 'insurance', 'bhxh', 'bhxhnn', 'bảo hiểm xã hội', 'bhxh', 'bhxhnn'],
    mota: 'Tạo bảo hiểm tháng'
  },
  
  // API Tạo Công Đoàn
  congdoan: {
    hanhdong: ['tạo', 'tao', 'thêm', 'them', 'tính', 'tinh', 'hạch toán', 'hach toan', 'xử lý', 'xu ly', 'sinh', 'generate'],
    doituong: ['công đoàn', 'cong doan', 'kinh phí công đoàn', 'kinh phi cong doan', 'kpcđ', 'kpcd', 'union', 'kinh phí công đoàn tháng'],
    mota: 'Tạo kinh phí công đoàn tháng'
  },
  
  // API Tạo CTGS
  ctgs: {
    hanhdong: ['tạo', 'tao', 'thêm', 'them', 'tính', 'tinh', 'hạch toán', 'hach toan', 'xử lý', 'xu ly', 'sinh', 'generate'],
    doituong: ['ctgs', 'chi tiết giao sổ', 'chi tiet giao so', 'giao sổ', 'giao so', 'giao so', 'chi tiết giao sổ mới'],
    mota: 'Tạo chi tiết giao sổ'
  }
};

// ===== PHẦN 2: HÀM KIỂM TRA TỪ KHÓA =====
function detectKeywordType(message) {
  const lowerMessage = message.toLowerCase().trim();
  
  console.log('🔍 Kiểm tra từ khóa trong tin nhắn:', lowerMessage);
  
  // Bước 1: Kiểm tra số CTGS (1-10) trước
  const ctgsIndex = parseInt(lowerMessage);
  if (!isNaN(ctgsIndex) && ctgsIndex >= 1 && ctgsIndex <= 10) {
    console.log(`🎯 Phát hiện chọn CTGS số ${ctgsIndex}`);
    return 'ctgs_selection';
  }
  
  // Bước 2: Kiểm tra từng API theo cấu trúc mới
  for (const [apiType, apiConfig] of Object.entries(API_KEYWORDS)) {
    const { hanhdong, doituong, mota } = apiConfig;
    
    // Kiểm tra có cả hành động và đối tượng không
    const hasAction = hanhdong.some(action => lowerMessage.includes(action));
    const hasObject = doituong.some(object => lowerMessage.includes(object));
    
    if (hasAction && hasObject) {
      console.log(`✅ Phát hiện API ${apiType}: ${mota}`);
      console.log(`   - Hành động: ${hanhdong.find(action => lowerMessage.includes(action))}`);
      console.log(`   - Đối tượng: ${doituong.find(object => lowerMessage.includes(object))}`);
      return apiType;
    }
    
    // Nếu chỉ có đối tượng (không cần hành động cụ thể)
    if (hasObject) {
      console.log(`✅ Phát hiện API ${apiType} (chỉ đối tượng): ${mota}`);
      console.log(`   - Đối tượng: ${doituong.find(object => lowerMessage.includes(object))}`);
      return apiType;
    }
  }
  
  console.log('💬 Không phát hiện từ khóa API');
  return null;
}

// ===== PHẦN 3: HÀM XỬ LÝ API THEO LOẠI =====
async function handleApiByType(type, userInfo, sessionId, message = '') {
  try {
    switch (type) {
      case 'luong':
        console.log('💰 Gọi API tạo lương...');
        return await taoLuongvoiCTGS(userInfo, sessionId);
        
      case 'baohiem':
        console.log('🛡️ Gọi API tạo bảo hiểm...');
        return await taoBaohiem(userInfo, sessionId);
        
      case 'congdoan':
        console.log('🏢 Gọi API tạo công đoàn...');
        return await taoCongdoan(userInfo, sessionId);
        
      case 'ctgs':
        console.log('📋 Gọi API tạo CTGS...');
        return await taoCTGS(userInfo, sessionId);
        
      case 'ctgs_selection':
        const ctgsIndex = parseInt(message);
        console.log(`🎯 Gọi API tạo lương với CTGS số ${ctgsIndex}...`);
        return await taoLuongvoiCTGSDachon(userInfo, ctgsIndex, sessionId);
        
      default:
        console.log('❌ Loại API không được hỗ trợ:', type);
        return {
          success: false,
          error: 'Loại API không được hỗ trợ'
        };
    }
  } catch (error) {
    console.error(`❌ Lỗi khi gọi API ${type}:`, error);
    return {
      success: false,
      error: error.message || 'Lỗi gọi API'
    };
  }
}

// ===== PHẦN 4: HÀM LƯU API CALL VÀO SESSION =====
async function luuApiCallVaoSession(sessionId, message, result, apiType) {
  try {
    const { ChatSessionService, ChatMessageService } = require('./services/databaseService');
    
    // Tạo nội dung response đơn giản
    let responseContent = '';
    if (result.success) {
      responseContent = result.message || result.data || 'Thao tác thành công';
    } else {
      responseContent = result.error || 'Có lỗi xảy ra';
    }
    
    // Lưu message của user vào database
    await ChatMessageService.create({
      session_id: sessionId,
      message_type: 'user',
      content: message,
      created_at: new Date()
    });
    
    // Lưu response của assistant vào database
    await ChatMessageService.create({
      session_id: sessionId,
      message_type: 'bot',
      content: responseContent,
      api_result: JSON.stringify({ 
        type: 'api_call',
        apiType: apiType,
        success: result.success,
        data: result.data || null,
        error: result.error || null
      }),
      created_at: new Date()
    });
    
    // Cập nhật thời gian hoạt động của session
    await ChatSessionService.updateLastActivity(sessionId);
    
    console.log(`💾 Đã lưu API call ${apiType} vào session ${sessionId}`);
    
  } catch (error) {
    console.error('❌ Lỗi khi lưu API call vào session:', error);
  }
}

// ===== PHẦN 5: HÀM KIỂM TRA Q&A DATABASE =====
async function checkQADatabase(message) {
  try {
    // Đảm bảo kết nối database
    const isConnected = await ensureConnection();
    if (!isConnected) {
      console.log('⚠️ Không thể kết nối database, sử dụng fallback');
      return { found: false, error: 'Database connection failed' };
    }

    const lowerMessage = message.toLowerCase().trim();
    console.log('🔍 Kiểm tra Q&A database cho tin nhắn:', lowerMessage);
    
    // Sử dụng QADatabaseService để tìm câu trả lời
    const qaResult = await QADatabaseService.findAnswer(message);
    
    if (qaResult) {
      console.log(`✅ Tìm thấy câu trả lời Q&A: ${qaResult.question} (matchScore: ${qaResult.matchScore?.toFixed(2)}%)`);
      return {
        found: true,
        question: qaResult.question,
        answer: qaResult.answer,
        category: qaResult.category,
        score: qaResult.score,
        matchScore: qaResult.matchScore
      };
    }
    
    console.log(`💬 Không tìm thấy câu trả lời phù hợp trong Q&A database`);
    return {
      found: false,
      score: 0
    };
    
  } catch (error) {
    console.error('❌ Lỗi khi kiểm tra Q&A database:', error);
    return {
      found: false,
      error: error.message
    };
  }
}

// ===== PHẦN 6: HÀM LƯU Q&A VÀO SESSION =====
async function luuQAVaoSession(sessionId, message, qaResult) {
  try {
    const { ChatSessionService, ChatMessageService } = require('./services/databaseService');
    
    // Lưu message của user vào database
    await ChatMessageService.create({
      session_id: sessionId,
      message_type: 'user',
      content: message,
      created_at: new Date()
    });
    
    // Lưu response của assistant vào database (chỉ lưu nội dung trả lời đơn giản)
    await ChatMessageService.create({
      session_id: sessionId,
      message_type: 'bot',
      content: qaResult.answer,
      api_result: JSON.stringify({ 
        type: 'qa_response',
        question: qaResult.question,
        category: qaResult.category,
        score: qaResult.score,
        matchScore: qaResult.matchScore
      }),
      created_at: new Date()
    });
    
    // Cập nhật thời gian hoạt động của session
    await ChatSessionService.updateLastActivity(sessionId);
    
    console.log(`💾 Đã lưu Q&A response vào session ${sessionId}`);
    
  } catch (error) {
    console.error('❌ Lỗi khi lưu Q&A vào session:', error);
  }
}

// ===== PHẦN 7: HÀM CHÍNH KIỂM TRA TỪ KHÓA =====
async function checkApiKeywords(message, userInfo, sessionId) {
  try {
    // Bước 1: Phát hiện loại từ khóa API
    const keywordType = detectKeywordType(message);
    
    // Bước 2: Nếu có từ khóa API, gọi API tương ứng
    if (keywordType) {
      const result = await handleApiByType(keywordType, userInfo, sessionId, message);
      
      // Lưu API call vào session
      await luuApiCallVaoSession(sessionId, message, result, keywordType);
      
      return {
        shouldCallApi: true,
        result: result
      };
    }
    
    // Bước 3: Kiểm tra Q&A database
    const qaResult = await checkQADatabase(message);
    if (qaResult.found) {
      // Lưu Q&A vào session
      await luuQAVaoSession(sessionId, message, qaResult);
      
      return {
        shouldCallApi: false,
        shouldUseQA: true,
        result: {
          success: true,
          message: qaResult.answer,
          source: 'qa_database',
          question: qaResult.question,
          category: qaResult.category,
          score: qaResult.score
        }
      };
    }
    
    // Bước 4: Không có từ khóa API và Q&A, chuyển sang ChatGPT
    return {
      shouldCallApi: false,
      shouldUseQA: false,
      result: null
    };
    
  } catch (error) {
    console.error('❌ Lỗi khi kiểm tra từ khóa API:', error);
    return {
      shouldCallApi: false,
      shouldUseQA: false,
      result: null
    };
  }
}

// Start server
app.listen(PORT, () => {
  console.log(`🚀 Server đang chạy trên port ${PORT}`);
  console.log(`📱 ChatBot GPT sẵn sàng phục vụ!`);
  console.log(`🌐 Truy cập: http://localhost:${PORT}`);
  console.log(`🗺️ Dynamic API Mapping đã được kích hoạt!`);
});

module.exports = app;
