const { ApiMapping, QADatabase, ChatSession, ChatMessage, ChatStatistic } = require('../models');
const fs = require('fs');
const path = require('path');

// API Mapping Service
class ApiMappingService {
  // Kiểm tra kết nối database
  static async checkConnection() {
    try {
      await ApiMapping.sequelize.authenticate();
      console.log('✅ Database connection OK');
      return true;
    } catch (error) {
      console.error('❌ Database connection failed:', error);
      return false;
    }
  }
  // Import từ JSON file
  static async importFromJson() {
    try {
      const jsonPath = path.join(__dirname, '../config/api-mapping.json');
      if (!fs.existsSync(jsonPath)) {
        console.log('❌ File api-mapping.json không tồn tại');
        return false;
      }

      const data = JSON.parse(fs.readFileSync(jsonPath, 'utf8'));
      
      // Đọc từ api_mapping object
      const apiMappings = data.api_mapping || data;
      
      for (const [domain, mapping] of Object.entries(apiMappings)) {
        await ApiMapping.upsert({
          domain: domain,
          api_server: mapping.api_server,
          environment: mapping.environment || 'development',
          description: mapping.description || '',
          is_active: true
        });
      }
      
      console.log('✅ Đã import API mappings vào database');
      return true;
    } catch (error) {
      console.error('❌ Lỗi import API mappings:', error);
      return false;
    }
  }

  // Lấy API mapping theo domain
  static async getByDomain(domain) {
    try {
      const mapping = await ApiMapping.findOne({
        where: { domain: domain, is_active: true }
      });
      return mapping;
    } catch (error) {
      console.error('❌ Lỗi lấy API mapping:', error);
      return null;
    }
  }

  // Lấy tất cả API mappings
  static async getAll() {
    try {
      return await ApiMapping.findAll({
        order: [['created_at', 'DESC']]
      });
    } catch (error) {
      console.error('❌ Lỗi lấy tất cả API mappings:', error);
      return [];
    }
  }

  // Lấy API mapping theo ID
  static async getById(id) {
    try {
      const mapping = await ApiMapping.findByPk(id);
      return mapping;
    } catch (error) {
      console.error('❌ Lỗi lấy API mapping theo ID:', error);
      return null;
    }
  }

  // Tạo API mapping mới
  static async create(data) {
    try {
      const mapping = await ApiMapping.create(data);
      return mapping;
    } catch (error) {
      console.error('❌ Lỗi tạo API mapping:', error);
      throw error;
    }
  }

  // Cập nhật API mapping
  static async update(id, data) {
    try {
      console.log('ApiMappingService.update called with:', { id, data });
      
      const mapping = await ApiMapping.findByPk(id);
      if (!mapping) {
        console.log('API mapping not found with ID:', id);
        return null;
      }
      
      // Validate data before update - allow all fields
      const updateData = {};
      const allowedFields = ['domain', 'api_server', 'environment', 'description', 'is_active', 'name', 'endpoint'];
      
      for (const field of allowedFields) {
        if (data[field] !== undefined && data[field] !== null) {
          updateData[field] = data[field];
        }
      }
      
      console.log('Updating with data:', updateData);
      
      if (Object.keys(updateData).length === 0) {
        console.log('No valid fields to update');
        return mapping;
      }
      
      // Kiểm tra xem domain có bị trùng không (nếu có cập nhật domain)
      if (updateData.domain && updateData.domain !== mapping.domain) {
        const existingMapping = await ApiMapping.findOne({
          where: { 
            domain: updateData.domain,
            id: { [require('sequelize').Op.ne]: id } // Loại trừ record hiện tại
          }
        });
        
        if (existingMapping) {
          throw new Error('Domain đã tồn tại trong hệ thống');
        }
      }
      
      await mapping.update(updateData);
      return mapping;
    } catch (error) {
      console.error('❌ Lỗi cập nhật API mapping:', error);
      throw error;
    }
  }

  // Xóa API mapping
  static async delete(id) {
    try {
      const mapping = await ApiMapping.findByPk(id);
      if (!mapping) {
        return false;
      }
      
      await mapping.destroy();
      return true;
    } catch (error) {
      console.error('❌ Lỗi xóa API mapping:', error);
      throw error;
    }
  }
}

// Q&A Database Service
class QADatabaseService {
  // Import từ JSON file
  static async importFromJson() {
    try {
      const jsonPath = path.join(__dirname, '../rag/kthc-qa-database.json');
      if (!fs.existsSync(jsonPath)) {
        console.log('❌ File kthc-qa-database.json không tồn tại');
        return false;
      }

      const data = JSON.parse(fs.readFileSync(jsonPath, 'utf8'));
      
      for (const qa of data.kthc_qa_database) {
        await QADatabase.upsert({
          question: qa.question,
          answer: qa.answer,
          category: qa.category || 'general',
          keywords: qa.keywords ? qa.keywords.join(',') : '',
          score: qa.score || 0,
          is_active: true
        });
      }
      
      console.log('✅ Đã import Q&A database vào database');
      return true;
    } catch (error) {
      console.error('❌ Lỗi import Q&A database:', error);
      return false;
    }
  }

  // Tìm câu trả lời với độ chính xác 60%
  static async findAnswer(message) {
    try {
      const qaList = await QADatabase.findAll({
        where: { is_active: true },
        order: [['score', 'DESC']]
      });

      let bestMatch = null;
      let bestScore = 0;
      const userWords = message.toLowerCase().trim().split(/\s+/).filter(word => word.length > 2);

      for (const qa of qaList) {
        let score = 0;
        const keywords = qa.keywords ? qa.keywords.split(',').map(k => k.trim()) : [];
        const questionWords = qa.question.toLowerCase().split(/\s+/).filter(word => word.length > 2);
        
        // 1. Kiểm tra câu hỏi chính xác (40% trọng số)
        if (message.toLowerCase().includes(qa.question.toLowerCase())) {
          score += 40;
        } else if (qa.question.toLowerCase().includes(message.toLowerCase())) {
          score += 30;
        }
        
        // 2. Kiểm tra từ khóa (30% trọng số)
        let keywordMatches = 0;
        for (const keyword of keywords) {
          if (message.toLowerCase().includes(keyword.toLowerCase())) {
            keywordMatches++;
          }
        }
        if (keywords.length > 0) {
          score += (keywordMatches / keywords.length) * 30;
        }
        
        // 3. Kiểm tra từ chung (20% trọng số)
        let commonWords = 0;
        for (const userWord of userWords) {
          if (questionWords.some(qWord => qWord.includes(userWord) || userWord.includes(qWord))) {
            commonWords++;
          }
        }
        if (userWords.length > 0) {
          score += (commonWords / userWords.length) * 20;
        }
        
        // 4. Kiểm tra độ dài tương đồng (10% trọng số)
        const lengthDiff = Math.abs(message.length - qa.question.length);
        const maxLength = Math.max(message.length, qa.question.length);
        if (maxLength > 0) {
          score += (1 - lengthDiff / maxLength) * 10;
        }

        console.log(`🔍 Q&A "${qa.question}" - Score: ${score.toFixed(2)}%`);
        
        if (score > bestScore) {
          bestScore = score;
          bestMatch = qa;
        }
      }

      // Ngưỡng 60% để trả về kết quả
      if (bestScore >= 60) {
        console.log(`✅ Tìm thấy Q&A phù hợp: "${bestMatch.question}" với độ chính xác ${bestScore.toFixed(2)}%`);
        return {
          ...bestMatch.toJSON(),
          matchScore: bestScore
        };
      } else {
        console.log(`❌ Không tìm thấy Q&A phù hợp. Điểm cao nhất: ${bestScore.toFixed(2)}%`);
        return null;
      }
    } catch (error) {
      console.error('❌ Lỗi tìm câu trả lời:', error);
      return null;
    }
  }

  // Lấy tất cả Q&A
  static async getAll() {
    try {
      return await QADatabase.findAll({
        order: [['created_at', 'DESC']]
      });
    } catch (error) {
      console.error('❌ Lỗi lấy tất cả Q&A:', error);
      return [];
    }
  }

  // Lấy Q&A theo ID
  static async getById(id) {
    try {
      const qa = await QADatabase.findByPk(id);
      return qa;
    } catch (error) {
      console.error('❌ Lỗi lấy Q&A theo ID:', error);
      return null;
    }
  }

  // Tạo Q&A mới
  static async create(data) {
    try {
      const qa = await QADatabase.create(data);
      return qa;
    } catch (error) {
      console.error('❌ Lỗi tạo Q&A:', error);
      throw error;
    }
  }

  // Cập nhật Q&A
  static async update(id, data) {
    try {
      const qa = await QADatabase.findByPk(id);
      if (!qa) {
        return null;
      }
      
      await qa.update(data);
      return qa;
    } catch (error) {
      console.error('❌ Lỗi cập nhật Q&A:', error);
      throw error;
    }
  }

  // Xóa Q&A
  static async delete(id) {
    try {
      const qa = await QADatabase.findByPk(id);
      if (!qa) {
        return false;
      }
      
      await qa.destroy();
      return true;
    } catch (error) {
      console.error('❌ Lỗi xóa Q&A:', error);
      throw error;
    }
  }
}

// Chat Session Service
class ChatSessionService {
  // Tạo session mới
  static async createSession(sessionId, userInfo = null, apiMapping = null) {
    try {
      const session = await ChatSession.create({
        session_id: sessionId,
        user_info: userInfo,
        api_mapping: apiMapping,
        last_activity: new Date()
      });
      return session;
    } catch (error) {
      console.error('❌ Lỗi tạo session:', error);
      return null;
    }
  }

  // Lấy session
  static async getSession(sessionId) {
    try {
      return await ChatSession.findOne({
        where: { session_id: sessionId, is_active: true }
      });
    } catch (error) {
      console.error('❌ Lỗi lấy session:', error);
      return null;
    }
  }

  // Lấy session theo session_id (alias cho getSession)
  static async getBySessionId(sessionId) {
    return await this.getSession(sessionId);
  }

  // Tạo session mới (alias cho createSession)
  static async create(sessionData) {
    try {
      const session = await ChatSession.create(sessionData);
      return session;
    } catch (error) {
      console.error('❌ Lỗi tạo session:', error);
      return null;
    }
  }

  // Cập nhật session
  static async update(id, updates) {
    try {
      const session = await ChatSession.findByPk(id);
      if (!session) {
        return false;
      }
      
      await session.update(updates);
      return true;
    } catch (error) {
      console.error('❌ Lỗi cập nhật session:', error);
      return false;
    }
  }

  // Cập nhật thời gian hoạt động
  static async updateLastActivity(sessionId) {
    try {
      await ChatSession.update(
        { last_activity: new Date() },
        { where: { session_id: sessionId } }
      );
      return true;
    } catch (error) {
      console.error('❌ Lỗi cập nhật last activity:', error);
      return false;
    }
  }

  // Cập nhật session
  static async updateSession(sessionId, updates) {
    try {
      return await ChatSession.update(updates, {
        where: { session_id: sessionId }
      });
    } catch (error) {
      console.error('❌ Lỗi cập nhật session:', error);
      return false;
    }
  }

  // Cập nhật thông tin session (user_info, api_mapping, last_activity)
  static async updateSessionInfo(sessionId, updates) {
    try {
      const result = await ChatSession.update(updates, {
        where: { session_id: sessionId }
      });
      return result[0] > 0; // Trả về true nếu có ít nhất 1 row được update
    } catch (error) {
      console.error('❌ Lỗi cập nhật session info:', error);
      return false;
    }
  }

  // Lưu message
  static async saveMessage(messageData) {
    try {
      const message = await ChatMessage.create(messageData);
      
      // Cập nhật last_activity của session
      await ChatSession.update(
        { last_activity: new Date() },
        { where: { session_id: messageData.session_id } }
      );
      
      return message;
    } catch (error) {
      console.error('❌ Lỗi lưu message:', error);
      return null;
    }
  }

  // Lấy lịch sử chat
  static async getChatHistory(sessionId, limit = 50) {
    try {
      const messages = await ChatMessage.findAll({
        where: { session_id: sessionId },
        order: [['created_at', 'ASC']],
        limit: limit
      });

      // Chuyển đổi message_type thành role
      return messages.map(message => {
        const messageData = message.toJSON();
        messageData.role = messageData.message_type === 'user' ? 'user' : 'assistant';
        return messageData;
      });
    } catch (error) {
      console.error('❌ Lỗi lấy lịch sử chat:', error);
      return [];
    }
  }

  // Lấy tất cả sessions
  static async getAll() {
    try {
      const sessions = await ChatSession.findAll({
        order: [['last_activity', 'DESC']],
        include: [{
          model: ChatMessage,
          attributes: ['id'],
          required: false
        }]
      });

      // Thêm message_count cho mỗi session
      return sessions.map(session => {
        const sessionData = session.toJSON();
        sessionData.message_count = session.ChatMessages ? session.ChatMessages.length : 0;
        delete sessionData.ChatMessages;
        return sessionData;
      });
    } catch (error) {
      console.error('❌ Lỗi lấy tất cả sessions:', error);
      return [];
    }
  }

  // Đếm sessions theo ngày
  static async getSessionCount(date) {
    try {
      return await ChatSession.count({
        where: {
          created_at: {
            [require('sequelize').Op.gte]: date
          }
        }
      });
    } catch (error) {
      console.error('❌ Lỗi đếm sessions:', error);
      return 0;
    }
  }

  // Đếm messages theo ngày
  static async getMessageCount(date) {
    try {
      return await ChatMessage.count({
        where: {
          created_at: {
            [require('sequelize').Op.gte]: date
          }
        }
      });
    } catch (error) {
      console.error('❌ Lỗi đếm messages:', error);
      return 0;
    }
  }

  // Xóa session và tất cả messages liên quan
  static async deleteSession(sessionId) {
    try {
      // Xóa tất cả messages của session trước
      await ChatMessage.destroy({
        where: { session_id: sessionId }
      });

      // Sau đó xóa session
      const result = await ChatSession.destroy({
        where: { session_id: sessionId }
      });

      return result > 0; // Trả về true nếu có session được xóa
    } catch (error) {
      console.error('❌ Lỗi xóa session:', error);
      return false;
    }
  }
}

// Statistics Service
class StatisticsService {
  // Cập nhật thống kê hàng ngày
  static async updateDailyStats() {
    try {
      const today = new Date().toISOString().split('T')[0];
      
      // Đếm sessions mới hôm nay
      const newSessions = await ChatSession.count({
        where: {
          created_at: {
            [require('sequelize').Op.gte]: today
          }
        }
      });

      // Đếm messages hôm nay
      const newMessages = await ChatMessage.count({
        where: {
          created_at: {
            [require('sequelize').Op.gte]: today
          }
        }
      });

      // Đếm API calls hôm nay
      const apiCalls = await ChatMessage.count({
        where: {
          message_type: 'api_call',
          created_at: {
            [require('sequelize').Op.gte]: today
          }
        }
      });

      // Đếm QA responses hôm nay
      const qaResponses = await ChatMessage.count({
        where: {
          message_type: 'qa_response',
          created_at: {
            [require('sequelize').Op.gte]: today
          }
        }
      });

      // Đếm GPT responses hôm nay
      const gptResponses = await ChatMessage.count({
        where: {
          message_type: 'bot',
          created_at: {
            [require('sequelize').Op.gte]: today
          }
        }
      });

      // Tính thời gian phản hồi trung bình
      const avgResponseTime = await ChatMessage.findOne({
        attributes: [
          [require('sequelize').fn('AVG', require('sequelize').col('processing_time')), 'avg_time']
        ],
        where: {
          created_at: {
            [require('sequelize').Op.gte]: today
          },
          processing_time: {
            [require('sequelize').Op.not]: null
          }
        }
      });

      // Upsert thống kê
      await ChatStatistic.upsert({
        date: today,
        total_sessions: newSessions,
        total_messages: newMessages,
        api_calls: apiCalls,
        qa_responses: qaResponses,
        gpt_responses: gptResponses,
        avg_response_time: avgResponseTime ? parseFloat(avgResponseTime.getDataValue('avg_time')) : 0,
        unique_users: newSessions // Đơn giản hóa
      });

      console.log('✅ Đã cập nhật thống kê ngày:', today);
    } catch (error) {
      console.error('❌ Lỗi cập nhật thống kê:', error);
    }
  }

  // Lấy thống kê
  static async getStats(days = 7) {
    try {
      const endDate = new Date();
      const startDate = new Date();
      startDate.setDate(startDate.getDate() - days);

      return await ChatStatistic.findAll({
        where: {
          date: {
            [require('sequelize').Op.between]: [startDate.toISOString().split('T')[0], endDate.toISOString().split('T')[0]]
          }
        },
        order: [['date', 'ASC']]
      });
    } catch (error) {
      console.error('❌ Lỗi lấy thống kê:', error);
      return [];
    }
  }
}

// Chat Message Service
class ChatMessageService {
  // Tạo message mới
  static async create(messageData) {
    try {
      const message = await ChatMessage.create(messageData);
      return message;
    } catch (error) {
      console.error('❌ Lỗi tạo message:', error);
      throw error;
    }
  }

  // Lấy message theo ID
  static async getById(id) {
    try {
      return await ChatMessage.findByPk(id);
    } catch (error) {
      console.error('❌ Lỗi lấy message:', error);
      return null;
    }
  }

  // Lấy messages theo session ID
  static async getBySessionId(sessionId, limit = 50) {
    try {
      return await ChatMessage.findAll({
        where: { session_id: sessionId },
        order: [['created_at', 'ASC']],
        limit: limit
      });
    } catch (error) {
      console.error('❌ Lỗi lấy messages theo session:', error);
      return [];
    }
  }

  // Cập nhật message
  static async update(id, updates) {
    try {
      const message = await ChatMessage.findByPk(id);
      if (!message) {
        return false;
      }
      
      await message.update(updates);
      return true;
    } catch (error) {
      console.error('❌ Lỗi cập nhật message:', error);
      return false;
    }
  }

  // Xóa message
  static async delete(id) {
    try {
      const message = await ChatMessage.findByPk(id);
      if (!message) {
        return false;
      }
      
      await message.destroy();
      return true;
    } catch (error) {
      console.error('❌ Lỗi xóa message:', error);
      return false;
    }
  }

  // Lấy tất cả messages
  static async getAll(limit = 100) {
    try {
      return await ChatMessage.findAll({
        order: [['created_at', 'DESC']],
        limit: limit
      });
    } catch (error) {
      console.error('❌ Lỗi lấy tất cả messages:', error);
      return [];
    }
  }
}

module.exports = {
  ApiMappingService,
  QADatabaseService,
  ChatSessionService,
  ChatMessageService,
  StatisticsService
};
